package com.xebialabs.deployit.deployment.orchestrator

import com.xebialabs.deployit.deployment.orchestrator.Descriptions._
import com.xebialabs.deployit.engine.spi.orchestration.{Orchestration, Orchestrations, Orchestrator}
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, Operation}
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.xlplatform._

import scala.collection.mutable
import scala.jdk.CollectionConverters._

trait DeployedOrchestratorBase extends Orchestrator with OrchestratorUtil {
  override def orchestrate(specification: DeltaSpecification): Orchestration = {
    val orderForOperation = getOrderForString(specification.getOperation)
    val deltas = specification.getDeltas
    val byDeployed = deltas.asScala.groupBy(_.correctDeployed).toList.sortBy(_._1.getName)(orderForOperation)
    val plan = byDeployed.map {
      case (deployed, delta: mutable.Buffer[Delta]) =>
        Orchestrations.interleaved(getDescriptionForDeployed(specification.getOperation, deployed, specification.getDeployedApplication), delta.toList.asJava)
    }
    compositeOrchestrator(DescriptionHelper.getDescriptionForSpec(specification), plan)
  }

  def compositeOrchestrator(description: String, plan: Seq[Orchestration]): Orchestration

}

@Orchestrator.Metadata(name = "parallel-by-deployed", description = "The parallel by deployed orchestrator.")
class ParallelByDeployed extends Orchestrator with DeployedOrchestratorBase {
  override def compositeOrchestrator(description: String, plan: Seq[Orchestration]): Orchestration = Orchestrations.parallel(description, plan.asJava)
}

@Orchestrator.Metadata(name = "sequential-by-deployed", description = "The sequential by deployed orchestrator.")
class SequentialByDeployed extends Orchestrator with DeployedOrchestratorBase {
  override def compositeOrchestrator(description: String, plan: Seq[Orchestration]): Orchestration = Orchestrations.serial(description, plan.asJava)
}

object Descriptions {
  private val deploymentVerbs = Map(Operation.CREATE -> "Deploy", Operation.DESTROY -> "Undeploy", Operation.MODIFY -> "Update", Operation.NOOP -> "Not update")
  private val provisioningVerbs = Map(Operation.CREATE -> "Provision", Operation.DESTROY -> "Deprovision", Operation.MODIFY -> "Update", Operation.NOOP -> "Not update")

  def nameOrNull(ci: ConfigurationItem): String = Option(ci).map(_.getName).orNull

  def verb(op: Operation, deployedApplication: DeployedApplication): String = {
    if (deployedApplication.getVersion.isInstanceOf[ProvisioningPackage]) {
      provisioningVerbs(op)
    } else {
      deploymentVerbs(op)
    }
  }

  def getDescriptionForDeployed(op: Operation, d: Deployed[_ <: Deployable, _ <: Container], deployedApplication: DeployedApplication): String = List(verb(op, deployedApplication), nameOrNull(d)).mkString(" ")
}
