package com.xebialabs.deployit.deployment
package rules

import com.xebialabs.deployit.plugin.api.deployment.planning.Checkpoint
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, Operation}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.rules.{Scope, StepPostConstructContext}
import com.xebialabs.platform.script.jython.JythonContext


private[rules] class XmlPlanRule(name: String,
                                 scope: Scope,
                                 override val condition: Option[String],
                                 steps: Iterable[XmlStepDataWithCheckPoint])
  extends XmlRule(name, scope, condition, steps) with PlanRule

private[rules] class XmlDeployedRule(name: String,
                                     override val ciTypes: Iterable[Type],
                                     override val operations: Iterable[Operation],
                                     override val condition: Option[String],
                                     steps: Iterable[XmlStepDataWithCheckPoint])
  extends XmlRule(name, Scope.DEPLOYED, condition, steps) with DeployedRule

private[rules] abstract class XmlRule(name: String, scope: Scope, condition: Option[String], steps: Iterable[XmlStepDataWithCheckPoint])
  extends Rule(name, scope) with ContextFactory {

  require(steps.forall(stepData => isUnique(stepData.stepParameters)), "Step parameters must be unique")

  def isUnique(steps: Iterable[XmlParameter]) = steps.map(_.name).toSet.size == steps.size

  override def doFire(scopedObject: AnyRef, planningContext: RulePlanningContext): Unit = {
    implicit val pContext = planningContext
    implicit val jContext: JythonContext = jythonContext(scopedObject, getScope)
    implicit val stepContext = stepPostConstructContext(scope, scopedObject)
    steps.foreach { stepData =>
      val step = createStep(stepData, planningContext.getStepRegistry, allBindings(scopedObject, getScope))
      stepData.checkpoint match {
        case Some(XmlCheckpoint(name, operation)) if scopedObject.isInstanceOf[Delta] =>
          val delta: Delta = scopedObject.asInstanceOf[Delta]
          val checkpoint: Checkpoint = new Checkpoint(delta, name.orNull, operation.orNull)
          planningContext.addStepWithCheckpoint(step, checkpoint)
        case None => planningContext.addStep(step)
        case _ => throw PlannerException(message = "Checkpoint was set for a scope that is not DEPLOYED")
      }
    }
  }

  private def createStep(data: XmlStepDataWithCheckPoint, stepRegistry: StepRegistry, bindings: Map[String, Object])(implicit jythonContext: JythonContext, stepContext: StepPostConstructContext): Step = {
    new StepFactory(stepRegistry, stepContext, new XmlParameterResolver(stepRegistry)).step(data.stepName, data.stepParameters.toSeq, bindings)
  }
}

sealed trait XmlParameter {
  val name: String
}

case class XmlExpressionParameter(name: String, expression: String) extends XmlParameter

case class XmlBasicParameter(name: String, value: String) extends XmlParameter

case class XmlCollectionParameter(name: String, parameters: Iterable[XmlParameter] = Nil) extends XmlParameter

case class XmlConfigurationItemParameter(name: String, ciType: String, fieldValues: Map[String, XmlConfigurationItemFieldParameter]) extends XmlParameter

case class XmlConfigurationItemFieldParameter(name: String, value: String, expression: Boolean = false)

sealed trait XmlStepElem

case class XmlCheckpoint(name: Option[String] = None, overrideOperation: Option[Operation] = None) extends XmlStepElem

case class XmlStepData(stepName: String, stepParameters: Iterable[XmlParameter] = Nil) extends XmlStepElem

object XmlStepDataWithCheckPoint {
  def apply(stepName: String, stepParameters: Iterable[XmlParameter] = Nil, checkpoint: Option[XmlCheckpoint] = None): XmlStepDataWithCheckPoint =
    XmlStepDataWithCheckPoint(XmlStepData(stepName, stepParameters), checkpoint)
}

case class XmlStepDataWithCheckPoint(stepData: XmlStepData, checkpoint: Option[XmlCheckpoint]) {
  def stepName = stepData.stepName

  def stepParameters = stepData.stepParameters
}