package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.plugin.api.deployment.specification.{DeltaSpecification, DeltaSpecificationWithDependencies, Operation}
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.api.udm
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployable
import com.xebialabs.deployit.repository.{ChangeSet, RepositoryService, RepositoryServiceHolder}

import scala.jdk.CollectionConverters._

class ContainersToEnvironmentContributor extends ChangeSetContributor {

  type DeployedContainer = BaseDeployedContainer[BaseDeployable, udm.Container]

  override def contribute(spec: DeltaSpecificationWithDependencies, changes: ChangeSet, ctx: ExecutionContext): Unit = {
    val specs = spec.getAllDeltaSpecifications.asScala.toSet
    contributeCreateOrUpdate(ctx, changes, specs = specs)
    contributeDelete(ctx, changes, specs = specs)
  }

  private[this] def contributeCreateOrUpdate(ctx: ExecutionContext, changes: ChangeSet, specs: Set[DeltaSpecification]): Unit = {
    val deployedsPerEnvironment = groupDeployedsByEnvironment(specs.filterNot(_.getOperation.equals(Operation.DESTROY)))
    doWithContainers(deployedsPerEnvironment, isCiInChangeset(_, changes)) { (env: Environment, containers: List[DeployedContainer]) =>
      env.getMembers.addAll(containers.asJava)
      if (ctx != null) {
        ctx.logOutput(s"Registering Containers ${containers.map(m => m.getName).sorted.mkString(", ")} to Environment ${env.getName}")
      }
      changes.createOrUpdate(env)
    }
  }

  private[this] def contributeDelete(ctx: ExecutionContext, changes: ChangeSet, specs: Set[DeltaSpecification]): Unit = {
    val deployedsPerEnvironment = groupDeployedsByEnvironmentFromPrevious(specs.filterNot(_.getOperation.equals(Operation.CREATE)))
    val deleteCiIds = changes.getDeleteCiIds.asScala.toSet
    doWithContainers(deployedsPerEnvironment, _ => true) { (env: Environment, containers: List[DeployedContainer]) =>
      env.getMembers.removeAll(containers.asJava)
      if (ctx != null) {
        ctx.logOutput(s"Removing Containers ${containers.map(m => m.getName).sorted.mkString(", ")} from Environment ${env.getName}")
      }
      changes.update(env)
      changes.delete(containers.filterNot(ci => deleteCiIds.contains(ci.getId)).asJava)
    }
  }

  def doWithContainers(deployedsPerEnvironment: Set[(Environment, List[Deployed[_ <: Deployable, _ <: Container]])],
                       filter: DeployedContainer => Boolean)(f: (Environment, List[DeployedContainer]) => Unit): Unit = {
    deployedsPerEnvironment.foreach { case (env, deployeds) =>
      val containers = deployeds.collect { case p: DeployedContainer => p }.filter(filter(_))
      if (containers.nonEmpty) {
        f(getEnvironment(env.getId), containers)
      }
    }
  }

  private[this] def getCiIds(ciList: List[ConfigurationItem]): List[String] = ciList.map(c => c.getId)

  private[this] def isCiInChangeset(deployed: Deployed[_ <: Deployable, _ <: Container], changes: ChangeSet): Boolean =
    getCiIds(changes.getCreateCis.asScala.toList).contains(deployed.getId) ||
      getCiIds(changes.getUpdateCis.asScala.toList).contains(deployed.getId) ||
      getCiIds(changes.getCreateOrUpdateCis.asScala.toList).contains(deployed.getId)

  private[this] def groupDeployedsByEnvironment(specs: Set[DeltaSpecification]): Set[(Environment, List[Deployed[_ <: Deployable, _ <: Container]])] =
    specs.map(d => d.getDeployedApplication.getEnvironment -> d.getDeployedApplication.getDeployeds.asScala.toList)

  private[this] def groupDeployedsByEnvironmentFromPrevious(specs: Set[DeltaSpecification]): Set[(Environment, List[Deployed[_ <: Deployable, _ <: Container]])] = {
    val list = specs.flatMap(s => s.getDeltas.asScala).filter(_.getOperation.equals(Operation.DESTROY)).map(_.getPrevious)
    specs.map(d => d.getPreviousDeployedApplication.getEnvironment -> d.getPreviousDeployedApplication.getDeployeds.asScala.toList.filter(list.contains))
  }

  private[this] def repositoryService: RepositoryService = RepositoryServiceHolder.getRepositoryService

  def getEnvironment(envId: String): Environment = repositoryService.read(envId)
}
