package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.satellite.SatelliteInfoChecker.Get
import com.xebialabs.deployit.plugin.satellite._
import com.xebialabs.deployit.tasksystem.TaskActorSystem
import com.xebialabs.satellite.protocol.Paths
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteGroup}
import org.apache.pekko.actor.{ActorRef, ActorSystem}

import java.util.StringJoiner
import scala.jdk.CollectionConverters._

trait Satellites {
  private[this] implicit lazy val actorSystem: ActorSystem = TaskActorSystem.actorSystem

  private[this] def preparingPhaseDescription(satellites: Set[Satellite]) = satelliteDescription("Preparing task on", satellites)

  private[this] def preparingPlanDescription(satellite: Satellite) = s"Prepare task for execution on ${satellite.getName}"

  private[this] def cleanUpPhaseDescription(satellites: Set[Satellite]) = satelliteDescription("Cleaning up", satellites)

  private[this] def satelliteDescription(verb: String, satellites: Set[Satellite], appendSatellite: Boolean = true) = if (satellites.isEmpty) "" else {
    val description = new StringJoiner(" ").add(verb)
    if (appendSatellite) {
      val s = if (satellites.size > 1) "s" else ""
      description.add(s"satellite$s")
    }
    description.add(satelliteNames(satellites))
    description.toString
  }

  private[this] def satelliteNames(satellites: Set[Satellite]): String = {
    satellites.map(_.getName).toList.sorted.mkString(", ")
  }

  private[this] def extractSatellitesFromPlans(promotedPlan: PhasedPlan): Set[Satellite] =
    promotedPlan.phases.asScala.flatMap(phase => extractSatellites(phase.plan)).toSet

  private[this] def extractSatellites(plan: ExecutablePlan): Set[Satellite] = plan match {
    case executablePlan: ExecutablePlan if executablePlan.satellite != null => Set(executablePlan.satellite)
    case _: StepPlan => Set.empty
    case composite: CompositePlan => composite.getSubPlans.asScala.flatMap(extractSatellites).toSet
  }

  private[this] def preparePhase(promotedPlan: PhasedPlan, allSatellites: Set[Satellite], cleanupCache: TaskOnSatelliteCleanupCache): PlanPhase = {
    implicit val uploader: ActorRef = SatelliteActors.uploader
    val prepareSatellitePlans = allSatellites.toList.sortBy(_.getName).map { sat =>
      val stepList: List[Step] = (sat match {
        case group: SatelliteGroup => List(ChooseSatelliteInGroupStep(group))
        case _ => Nil
      }) ::: (CheckExtensionsStep(sat) :: SendToSatelliteStep(sat, cleanupCache) :: Nil)

      new StepPlan(preparingPlanDescription(sat), stepList.asJava, promotedPlan.getListeners)
    }

    lazy val parallelPlan = new ParallelPlan(
      preparingPhaseDescription(allSatellites),
      prepareSatellitePlans.asJava,
      promotedPlan.getListeners
    )
    val preparingPlan = if (allSatellites.size == 1) prepareSatellitePlans.headOption.getOrElse(parallelPlan) else parallelPlan

    new PlanPhase(preparingPlan, preparingPhaseDescription(allSatellites), promotedPlan.getListeners)
  }

  private[this] def cleanUpSatellitePhase(promotedPlan: PhasedPlan, allSatellites: Set[Satellite], cleanupCache: TaskOnSatelliteCleanupCache): PlanPhase = {
    val steps: Set[Step] = allSatellites.map(CleanUpSatelliteStep(_, cleanupCache))
    val phaseDescription: String = cleanUpPhaseDescription(allSatellites)
    val cleanupPlan = new StepPlan(phaseDescription, steps.asJava, promotedPlan.getListeners)
    new PlanPhase(cleanupPlan, phaseDescription, promotedPlan.getListeners, true)
  }

  def prepareForSatelliteExecution(plan: PhasedPlan): PhasedPlan = {
    val satellites = extractSatellitesFromPlans(plan)

    if (satellites.nonEmpty) {
      satellites
        .flatMap(_.collectSatellites())
        .foreach { satellite =>
          val actorLocator = ActorLocator(satellite)
          val checker = actorSystem.actorOf(SatelliteInfoChecker.props(satellite.getAddress))
          checker ! Get(actorLocator.locate(Paths.info))
        }

      val cleanupCache = new TaskOnSatelliteCleanupCache
      plan.copy(phases = ((preparePhase(plan, satellites, cleanupCache) :: plan.phases.asScala.toList) :+ cleanUpSatellitePhase(plan, satellites, cleanupCache)).asJava)
    } else {
      plan
    }
  }

}

object Satellites extends Satellites
