package com.xebialabs.deployit.deployment.rules

import com.google.common.collect.Lists._
import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.deployment.rules.RegistriesActor.Messages.{RuleRegistryGet, RuleRegistryReturn, StepRegistryGet, StepRegistryReturn}
import com.xebialabs.deployit.deployment.service.ArtifactTransformerFactory
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification
import com.xebialabs.deployit.plugin.api.services.Repository
import org.apache.pekko.actor.{ActorRef, ActorSystem}
import org.apache.pekko.dispatch.MessageDispatcher
import org.apache.pekko.pattern.ask
import org.apache.pekko.util.Timeout

import java.util
import java.util.concurrent.TimeUnit
import scala.concurrent.Await
import scala.concurrent.duration.FiniteDuration
import scala.jdk.CollectionConverters._

class PlanCreationContextFactory(registriesActor: ActorRef, actorSystem: ActorSystem) {

  private implicit val timeout: Timeout = FiniteDuration(actorSystem.settings.config.getDuration("deploy.task.planner.registries.timeout", TimeUnit.SECONDS), TimeUnit.SECONDS)
  private implicit val executionContext: MessageDispatcher = actorSystem.dispatchers.lookup("deploy.task.planner.registries.Dispatcher")

  def createPlanCreationContext(repository: Repository, artifactTransformerFactory: ArtifactTransformerFactory): PlanCreationContext = {
    val ruleResult = (registriesActor ? RuleRegistryGet()).mapTo[RuleRegistryReturn].map(_.ruleRegistry)
    val stepResult = (registriesActor ? StepRegistryGet()).mapTo[StepRegistryReturn].map(_.stepRegistry)
    implicit val globalContext = new GlobalDeploymentPlanningContext
    implicit val stepRegistry: StepRegistry = Await.result(stepResult, timeout.duration)
    implicit val ruleRegistry: RuleRegistry = Await.result(ruleResult, timeout.duration)
    new PlanCreationContext(stepRegistry, repository, ruleRegistry, artifactTransformerFactory, globalContext)
  }
}

class PlanCreationContext(val stepRegistry: StepRegistry, val repository: Repository, val ruleRegistry: RuleRegistry,
                          val stitchEngineFactory: ArtifactTransformerFactory, val globalContext: GlobalDeploymentPlanningContext) {

  private val listenerAccumulator: util.List[ExecutionStateListener] = new util.ArrayList[ExecutionStateListener]()

  def stepPlan(description: String) = new StepPlan(description, listenerAccumulator)

  def phasedPlan(stepPlans: List[ExecutablePlan], description: String): PhasedPlan = {
    val serialPlan = new SerialPlan(description, stepPlans.asJava, listenerAccumulator)
    val planPhase: PlanPhase = new PlanPhase(serialPlan, "Deploy", listenerAccumulator)
    new PhasedPlan(newArrayList(planPhase), listenerAccumulator)
  }

  def parallelPlan(description: String, plans: List[ExecutablePlan]): ParallelPlan =
    new ParallelPlan(description, plans.asJava, listenerAccumulator)

  def serialPlan(description: String, plans: List[ExecutablePlan]): SerialPlan =
    new SerialPlan(description, plans.asJava, listenerAccumulator)

  def planningContext(plan: StepPlan, deltaSpec: DeltaSpecification): RulePlanningContext =
    RulePlanningContext(deltaSpec, stepRegistry, repository, stitchEngineFactory, plan, false, globalContext)

}
