package com.xebialabs.deployit.deployment.orchestrator;

import com.google.common.collect.Iterables;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.engine.spi.orchestration.Orchestrator;
import nl.javadude.scannit.Scannit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;

public final class OrchestratorRegistry {

    private OrchestratorRegistry() {}

    private static final Logger logger = LoggerFactory.getLogger(OrchestratorRegistry.class);
    private static Map<String, Orchestrator> lookup = newHashMap();

    static {
        Set<Class<? extends Orchestrator>> orchestratorClasses = Scannit.getInstance().getSubTypesOf(Orchestrator.class);
        for (Class<? extends Orchestrator> orchestratorClass : orchestratorClasses) {
            try {
                register(orchestratorClass);
            } catch (Exception e) {
                throw new RuntimeException("Couldn't boot the OrchestratorRegistry", e);
            }
        }
    }

    private static void register(Class<? extends Orchestrator> orchestratorClass) throws IllegalAccessException, InstantiationException {
        if (Modifier.isAbstract(orchestratorClass.getModifiers())) return;

        String name = orchestratorClass.getSimpleName();
        if (orchestratorClass.isAnnotationPresent(Orchestrator.Metadata.class)) {
            Orchestrator.Metadata annotation = orchestratorClass.getAnnotation(Orchestrator.Metadata.class);
            name = annotation.name();
        }
        logger.info("Registering Orchestrator [{}] under name [{}]", orchestratorClass, name);
        lookup.put(name, orchestratorClass.newInstance());
    }

    public static Orchestrator getOrchestrator(String name) {
        Orchestrator orchestrator = lookup.get(name);
        if(orchestrator == null){
            throw new OrchestratorRegistryException(String.format("Could not find a registered orchestrator with name [%s]", name));
        }
        return orchestrator;
    }

    public static List<Orchestrator> getOrchestrators(List<String> names) {
        return newArrayList(Iterables.transform(names, (final String input) -> getOrchestrator(input)));
    }

    public static Set<String> all() {
        return newHashSet(lookup.keySet());
    }

    @HttpResponseCodeResult(statusCode = 400)
    private static class OrchestratorRegistryException extends DeployitException {
        public OrchestratorRegistryException(String message) {
            super(message);
        }
    }
}
