package com.xebialabs.deployit.deployment.rules

import com.xebialabs.deployit.deployment.orchestrator.{OrchestratorComposer, OrchestratorRegistry}
import com.xebialabs.deployit.deployment.planner.PipedPlanner.PlanProducer
import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.deployment.rules.Implicits._
import com.xebialabs.deployit.engine.spi.exception.{DeployitException, HttpResponseCodeResult}
import com.xebialabs.deployit.engine.spi.orchestration._
import com.xebialabs.deployit.plugin.api.deployment.specification.{Delta, DeltaSpecification, DeltaSpecificationWithDependencies}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.rules.Scope
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._
import scala.collection.mutable

class RuleBasedPlanner extends PlanProducer with Logging with RulePlannerHelper {
  private val orchestrator = new OrchestratorComposer()

  override def produce(spec: MultiDeltaSpecification, planCreationContext: PlanCreationContext): PhasedPlan =
    try {
      createPlan(spec, planCreationContext)
    } catch {
      case pe: PlannerException =>
        throw pe
      case ex: Exception =>
        logger.error("", ex)
        throw PlannerException(s"An error occurred while planning: ${ex.getMessage}", ex)
    }

  private[rules] def createPlan(spec: MultiDeltaSpecification, planCreationContext: PlanCreationContext): PhasedPlan = {
    val prePlan = createPrePostPlan(spec, planCreationContext, "Prepare", Scope.PRE_PLAN)
    val orchestration = createOrchestrations(spec)
    val plan = createPlanOfScopeDeployedAndPlan(spec, planCreationContext)(orchestration)
    val postPlan = createPrePostPlan(spec, planCreationContext, "Finalize", Scope.POST_PLAN)
    planCreationContext.phasedPlan(List(prePlan, plan, postPlan), orchestration.getDescription)
  }

  private def createOrchestrations(spec: MultiDeltaSpecification): Orchestration = {
    val orchestratorIds = spec.getOrchestrators
    val orchestrators = OrchestratorRegistry.getOrchestrators(orchestratorIds)
    orchestrator.orchestrate(orchestrators.asScala.toList, spec)
  }

  private def createPrePostPlan(spec: DeltaSpecificationWithDependencies,
                                planCreationContext: PlanCreationContext,
                                blockDescription: String,
                                scope: Scope): ExecutablePlan =
    spec
      .getAllDeltaSpecifications
      .asScala
      .map(stepPlanOfScopePrePostPlan(_, planCreationContext, scope))
      .foldLeft(planCreationContext.stepPlan(blockDescription))(_ merge _)

  private def createPlanOfScopeDeployedAndPlan(spec: DeltaSpecificationWithDependencies, planCreationContext: PlanCreationContext)
                                              (orchestration: Orchestration): ExecutablePlan = orchestration match {
    case pp: ParallelOrchestration =>
      planCreationContext.parallelPlan(orchestration.getDescription, pp.getPlans.asScala.toList.map(createPlanOfScopeDeployedAndPlan(spec, planCreationContext)))
    case sp: SerialOrchestration =>
      planCreationContext.serialPlan(orchestration.getDescription, sp.getPlans.asScala.toList.map(createPlanOfScopeDeployedAndPlan(spec, planCreationContext)))
    case ip: InterleavedOrchestration =>
      createInterleavedPlan(ip, spec, planCreationContext)
  }

  private def createInterleavedPlan(orchestration: InterleavedOrchestration,
                                    spec: DeltaSpecificationWithDependencies,
                                    planCreationContext: PlanCreationContext): StepPlan = {
    val mainSpec = spec.getAllDeltaSpecifications.asScala.lastOption.orNull
    // Associate all the delta's in this orchestration with their original DeltaSpecification. If there is no spec associated, due to an injected delta,
    // associate it with the main spec, which is last in the list of allDeltaSpecifications
    val deltasToSpecs = spec.getAllDeltaSpecifications.asScala.flatMap(spec => spec.getDeltas.asScala.map(delta => delta -> spec)).toMap

    val by: Map[DeltaSpecification, mutable.Buffer[Delta]] = orchestration
      .getDeltas
      .asScala
      .groupBy(deltasToSpecs.getOrElse(_, mainSpec))

    val stepPlans: Iterable[StepPlan] = spec
      .getAllDeltaSpecifications
      .asScala
      .flatMap(spec => by.get(spec).map(deltas => createInterleavedPlan(deltas.toSeq, spec, planCreationContext)))

    stepPlans.foldLeft(planCreationContext.stepPlan(orchestration.getDescription))(_ merge _)
  }

  private def createInterleavedPlan(deltas: Seq[Delta], spec: DeltaSpecification, planCreationContext: PlanCreationContext): StepPlan = {
    val stepPlan = planCreationContext.stepPlan("Step Plan")
    val ctx = planCreationContext.planningContext(stepPlan, spec)
    addStepsOfScopeDeployed(deltas, stepPlan, ctx, planCreationContext.ruleRegistry)
    addStepsOfPlanScope(deltas, stepPlan, ctx, planCreationContext.ruleRegistry)
    addImplicitCheckpoints(spec, planCreationContext, stepPlan)
    stepPlan
  }

  private def addImplicitCheckpoints(spec: DeltaSpecification, planCreationContext: PlanCreationContext, stepPlan: StepPlan): Unit = {
    val ctx = planCreationContext.planningContext(stepPlan, spec)
    val checkpointedDeltas = stepPlan.getCheckpoints.asScala.map(_.getDelta).toSet
    stepPlan
      .getStepsWithPlanningInfo
      .asScala
      .flatMap { stepWithInfo => stepWithInfo.getDeltas.asScala.withFilter(!checkpointedDeltas.contains(_)).map(_ -> stepWithInfo.getStep) }
      .groupBy { case (delta, _) => delta }
      .foreach { case (delta, steps: mutable.Buffer[(Delta, Step)]) =>
        val uniqueOrderedSteps = mutable.LinkedHashSet(steps.toList: _*)
        val ((_, lastStep), _) = uniqueOrderedSteps.zipWithIndex
          .maxByOption { case ((_, step), index) => step.getOrder -> index }
          .getOrElse(throw new NoSuchElementException("Cannot get last step"))
        ctx.addCheckpoint(lastStep, delta)
      }
  }
}

@HttpResponseCodeResult(statusCode = 400)
case class PlannerException(message: String = null, cause: Throwable = null) extends DeployitException(message, cause)
