package com.xebialabs.deployit.deployment.rules

import javax.annotation.PreDestroy
import akka.actor.{ActorSystem, Terminated}
import com.typesafe.config.Config
import com.xebialabs.deployit.deployment.planner._
import com.xebialabs.deployit.deployment.stager.DeploymentStager
import com.xebialabs.xlplatform.config.{ConfigLoader, ConfigurationHolder}
import com.xebialabs.xlplatform.settings.CommonSettings
import com.xebialabs.xlplatform.watcher.FileWatchActor
import grizzled.slf4j.Logging

import scala.concurrent.Await

class RuleBasedPlannerFactory(val system: ActorSystem) extends Logging {
  private lazy val fileWatcherActor = system.actorOf(FileWatchActor.props, "file-watch-actor")
  private lazy val registriesActor = system.actorOf(RegistriesActor.props(fileWatcherActor), "registriesActor")

  def this() = this({
    val xlConfig: Config = ConfigLoader.loadWithDynamic(ConfigurationHolder.get())
    val actorConfig = xlConfig.getConfig("deploy.task.planner.system")
      .withValue("xl.file-watch.interval", xlConfig.getValue("deploy.task.planner.file-watch.interval"))
      .withValue("deploy.task", xlConfig.getValue("deploy.task"))
    ActorSystem("PlannerActorSystem", actorConfig)
  })

  val planner: Planner = PipedPlanner.performanceAware(
    new RuleBasedPlanner,
    Seq(
      new SimplifyingDeploymentPlanner,
      new FilterEmptyPhasePlanner,
      new OrderChunkingPlanner,
      new SizeChunkingPlanner,
      new RepositoryUpdateStepPlanner,
      new DeploymentStager,
      new SatelliteDeploymentPlanner
    )
  )

  val plannerWithoutStaging: Planner = PipedPlanner.performanceAware(
    new RuleBasedPlanner,
    Seq(
      new SimplifyingDeploymentPlanner,
      new FilterEmptyPhasePlanner,
      new OrderChunkingPlanner,
      new SizeChunkingPlanner,
      new RepositoryUpdateStepPlanner,
      new SatelliteDeploymentPlanner
    )
  )

  def planerCreationContextFactory: PlanCreationContextFactory = new PlanCreationContextFactory(registriesActor, system)

  @PreDestroy
  def shutdownTheSystem(): Terminated = {
    debug("Shutting down the planning actor system...")
    val shutdownTimeout = CommonSettings(system).tasker.shutdownTimeout
    Await.result(system.terminate(), shutdownTimeout)
  }
}
