package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.PipedPlanner.{PlanTransformer, TransformerContext}
import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.deployment.planner.StepPlan.StepWithPlanningInfo
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteAware}

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions

class SatelliteDeploymentPlanner extends PlanTransformer with PlanSugar {
  override def transform(context: TransformerContext): PhasedPlan = {
    val promotedPlan = promotePlan(context.plan)
    Satellites.prepareForSatelliteExecution(promotedPlan)
  }

  def promotePlan(satellitePlan: Plan): PhasedPlan = satellitePlan match {
    case phasedPlan: PhasedPlan => promoteSatellite(phasedPlan)
    case planPhase: PlanPhase => new PhasedPlan(List(promoteSatellite(planPhase)).asJava, planPhase.getListeners)
    case plan: ExecutablePlan => new PhasedPlan(List(new PlanPhase(promoteSatellite(plan), plan.getDescription, plan.getListeners)).asJava, plan.getListeners)
  }

  private def promoteSatellite(plan: ExecutablePlan): ExecutablePlan = plan match {
    case stepPlan: StepPlan => promoteSatellite(stepPlan)
    case serialPlan: SerialPlan => promoteSatellite(serialPlan)
    case parallelPlan: ParallelPlan => promoteSatellite(parallelPlan)
  }

  private def promoteSatellite(phasedPlan: PhasedPlan): PhasedPlan =
    phasedPlan.copy(phases = phasedPlan.phases.asScala.map(promoteSatellite).asJava)

  private def promoteSatellite(planPhase: PlanPhase): PlanPhase =
    planPhase.copy(plan = promoteSatellite(planPhase.plan))

  private def promoteSatellite(stepPlan: StepPlan): ExecutablePlan = {
    val result = stepPlan
      .getStepsWithPlanningInfo
      .asScala
      .foldLeft(List[(Satellite, ListBuffer[StepWithPlanningInfo])]()) { (tuples, step) =>
        val satellite = getSatelliteOrNull(step.getStep)
        tuples match {
          case Nil => List((satellite, ListBuffer(step)))
          case (currSat, buff) :: _ if currSat == satellite =>
            buff.append(step)
            tuples
          case _ => (satellite, ListBuffer(step)) :: tuples
        }
      }
    buildStepPlanResult(result, stepPlan)
  }

  private def buildStepPlanResult(result: List[(Satellite, ListBuffer[StepWithPlanningInfo])],
                                  originalPlan: StepPlan): ExecutablePlan =
    result match {
      case List((null, _)) => originalPlan
      case List((sat, _)) =>
        originalPlan.copy(satellite = sat, description = originalPlan.getDescription + s" with ${sat.getName}")
      case _ =>
        val stepPlans: List[ExecutablePlan] = result.reverse.map {
          case (sat, swpi) if sat == null => originalPlan.copy(steps = swpi.asJava, satellite = sat)
          case (sat, swpi) => originalPlan.copy(originalPlan.getDescription + s" ${
            sat.getName
          }", swpi.asJava, findCheckpoints(swpi, originalPlan.getCheckpoints), satellite = sat)
        }
        originalPlan.toSerial(subPlans = stepPlans.asJava)
    }

  private def promoteSatellite(plan: CompositePlan): ExecutablePlan = {
    val modifiedSubPlans = plan.getSubPlans.asScala.map(promoteSatellite)
    val satellites = modifiedSubPlans.map(_.satellite).toSet
    val promotedSatellite = satellites.toList match {
      case Nil => null
      case satellite :: Nil => satellite
      case _ :: tail if tail.nonEmpty => null
    }
    plan.copy(subPlans = modifiedSubPlans.asJava, satellite = promotedSatellite)
  }

  private def getSatelliteOrNull(step: Step): Satellite = step match {
    case stepSat: SatelliteAware => stepSat.getSatellite
    case _ => null
  }
}
