package com.xebialabs.deployit.deployment.planner


import com.xebialabs.deployit.deployment.ChangeSetBuilder
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState._
import com.xebialabs.deployit.plugin.api.deployment.specification.{DeltaSpecificationWithDependencies, Operation}
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import com.xebialabs.deployit.repository._
import com.xebialabs.deployit.repository.placeholders.{PlaceholderRepositoryHolder, ResolvedPlaceholderRepository}

import scala.beans.BeanProperty

class RepositoryUpdateStep(@BeanProperty val order: Int = 100,
                           @BeanProperty var description: String = "Register deployeds",
                           @BeanProperty val spec: DeltaSpecificationWithDependencies,
                           @BeanProperty val contributors: List[ChangeSetContributor] = Nil) extends Step {

  override def execute(ctx: ExecutionContext): StepExitCode = {

    ctx.logOutput("Starting repository update.")
    val changes: ChangeSet = determineChanges(ctx)
    contributors.foreach(c => c.contribute(spec, changes, ctx))
    logRepositoryUpdateActions(ctx, changes)
    RepositoryServiceHolder.getRepositoryService.execute(changes, new RepositoryUpdateStepProgressLogger(ctx))
    spec.getAllDeltaSpecifications.forEach { d =>
      handleResolvedPlaceHolders(d.getOperation, d.getDeployedApplication, ctx.getTask.getId, PlaceholderRepositoryHolder.getResolvedPlaceholderRepository)
    }
    ctx.logOutput("Finished repository update.")
    StepExitCode.SUCCESS
  }

  private def handleResolvedPlaceHolders(operation: Operation, deployedApplication: DeployedApplication, taskId: String, resolvedPlaceholderRepository: ResolvedPlaceholderRepository): Unit = {
    operation match {
      case Operation.CREATE =>
        resolvedPlaceholderRepository.saveResolvedPlaceholders(taskId, deployedApplication.get$ResolvedPlaceholders)
      case Operation.MODIFY =>
        resolvedPlaceholderRepository.archiveResolvedPlaceholder(deployedApplication.getId)
        resolvedPlaceholderRepository.saveResolvedPlaceholders(taskId, deployedApplication.get$ResolvedPlaceholders)
      case Operation.DESTROY =>
        resolvedPlaceholderRepository.archiveResolvedPlaceholder(deployedApplication.getId)
      case Operation.NOOP => //nothing
    }
  }

  def determineChanges(ctx: ExecutionContext): ChangeSet = {
    val checkPointManagerListener: CheckPointManagerListener = getCheckpointManagerListener(ctx)
    checkPointManagerListener.getCurrentTaskState match {
      case CANCELLING => checkPointManagerListener.checkpointManager.changeSet
      case _ => ChangeSetBuilder.determineChanges(spec)
    }
  }

  private def logRepositoryUpdateActions(ctx: ExecutionContext, changes: ChangeSet): Unit = {
    if (changes.getCreateCis.isEmpty && changes.getUpdateCis.isEmpty && changes.getDeleteCiIds.isEmpty) {
      ctx.logOutput("No items to update")
    }
  }

  private def getCheckpointManagerListener(context: ExecutionContext): CheckPointManagerListener = {
    val name: String = classOf[CheckPointManagerListener].getName
    context.getAttribute(name).asInstanceOf[CheckPointManagerListener]
  }

  override def skippable = false
}


class RepositoryUpdateStepProgressLogger(ctx: ExecutionContext) extends ProgressLogger {

  override def log(output: String, params: AnyRef*) = ctx.logOutput(String.format(output, params.toArray: _*))

}
