package com.xebialabs.deployit.deployment.planner

import com.xebialabs.deployit.deployment.planner.OrderChunkingPlanner.{orderDescription, threshold}
import com.xebialabs.deployit.deployment.planner.PipedPlanner.{PlanTransformer, TransformerContext}
import com.xebialabs.deployit.deployment.planner.PlanSugar._
import com.xebialabs.deployit.deployment.planner.StepPlan.StepWithPlanningInfo

import scala.jdk.CollectionConverters._
import scala.collection.mutable

object OrderChunkingPlanner {
  val threshold: Int = 30

  val orderDescription = Map(
    0 -> "Pre flight",
    1 -> "Stop artifacts",
    2 -> "Stop containers",
    3 -> "Undeploy artifacts",
    4 -> "Destroy resources",
    5 -> "In between",
    6 -> "Create resources",
    7 -> "Deploy artifacts",
    8 -> "Start containers",
    9 -> "Start artifacts",
    10 -> "Post flight"
  )
}

class OrderChunkingPlanner extends PlanTransformer with PlanSugar {
  def transform(context: TransformerContext): PhasedPlan = chunkByOrder(context.plan)

  def chunkByOrder(complexPlan: Plan): PhasedPlan = complexPlan match {
    case phasedPlan: PhasedPlan => chunkByOrder(phasedPlan)
    case planPhase: PlanPhase => new PhasedPlan(List(chunkByOrder(planPhase)).asJava, planPhase.getListeners)
    case plan: ExecutablePlan => new PhasedPlan(List(new PlanPhase(chunkByOrder(plan), plan.getDescription, plan.getListeners)).asJava, plan.getListeners)
  }

  def chunkByOrder(plan: PhasedPlan): PhasedPlan = plan.copy(phases = plan.phases.asScala.map(chunkByOrder).asJava)

  def chunkByOrder(planPhase: PlanPhase): PlanPhase = planPhase.copy(plan = chunkByOrder(planPhase.plan))

  def chunkByOrder(plan: ExecutablePlan): ExecutablePlan = plan match {
    case cp: CompositePlan => cp.copy(subPlans = cp.getSubPlans.asScala.map(chunkByOrder).asJava)
    case stp: StepPlan if stp.getStepsSize > threshold =>
      val chunks: List[ExecutablePlan] = chunkByOrder(stp)
      if (chunks.length <= 1)
        stp
      else
        stp.toSerial(subPlans = chunks.asJava)
    case stp: StepPlan => stp
  }

  def chunkByOrder(plan: StepPlan): List[StepPlan] = {
    plan
      .getStepsWithPlanningInfo
      .asScala
      .groupBy(_.getStep.getOrder / 10).toList
      .sortBy { case (order, _) => order }
      .map { case (order, steps) => createStepPlanForOrderedChunks(order, steps, plan) }
  }

  def createStepPlanForOrderedChunks(order: Int, steps: mutable.Buffer[StepWithPlanningInfo], plan: StepPlan): StepPlan =
    plan.copy(
      description = s"Steps with order ${orderDescription.getOrElse(order, "Undefined")}",
      steps = steps.asJava,
      checkpoints = findCheckpoints(steps, plan.getCheckpoints)
    )
}
