package com.xebialabs.deployit.deployment.planner;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.xebialabs.deployit.deployment.service.ArtifactTransformerFactory;
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.planning.Checkpoint;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.services.ArtifactTransformer;
import com.xebialabs.deployit.plugin.api.services.ContextInjectingArtifactTransformer;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

public class DefaultDeploymentPlanningContext implements DeploymentPlanningContext {
    public static final String DEPLOYMENT_CONTEXT_KEY = "deploymentContext";
    public static final String DC_DEPLOYED_APPLICATION_KEY = "deployedApplication";
    public static final String DC_PREVIOUS_DEPLOYED_APPLICATION_KEY = "previousDeployedApplication";
    public static final String DC_PLAN_KEY = "plan";
    public static final String DC_ROLLBACK_FLAG_KEY = "rollback";

    private static final String PATCH_TRANSFORMER_NAME = "patchDictionaryTransformer";
    private static final Function<DeploymentStep, Step> adapt = (DeploymentStep input) -> StepAdapter.wrapIfNeeded(input);

    private final HashMap<String, Object> attributes = newHashMap();
    private DeployedApplication deployedApplication;
    private DeployedApplication previousDeployedApplication;
    private Repository readOnlyRepository;
    private ArtifactTransformerFactory artifactTransformerFactory;
    private StepPlan plan;
    private boolean rollback;

    public DefaultDeploymentPlanningContext(DeployedApplication deployedApplication, DeployedApplication previousDeployedApplication,
                                            Repository readOnlyRepository, ArtifactTransformerFactory artifactTransformerFactory, StepPlan plan, boolean rollback) {
        this.deployedApplication = deployedApplication;
        this.previousDeployedApplication = previousDeployedApplication;
        this.readOnlyRepository = readOnlyRepository;
        this.plan = plan;
        this.rollback = rollback;
        this.artifactTransformerFactory = artifactTransformerFactory;
    }

    @Override
    public void addStep(DeploymentStep step) {
        addStep(adapt.apply(step));
    }

    @Override
    public void addSteps(DeploymentStep... steps) {
        addSteps(Iterables.transform(newArrayList(steps), adapt));
    }

    @Override
    public void addSteps(Collection<DeploymentStep> steps) {
        addSteps(Iterables.transform(steps, adapt));
    }

    @Override
    public void addStep(Step step) {
        logger.trace("Adding step [{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        plan.addStep(step);
    }

    @Override
    public void addSteps(Step... steps) {
        logger.trace("Adding steps");
        for (Step step : steps) {
            logger.trace("[{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        }
        plan.addSteps(newArrayList(steps));
    }

    @Override
    public void addSteps(Iterable<Step> steps) {
        logger.trace("Adding steps");
        for (Step step : steps) {
            logger.trace("[{} - {} ({})]", step.getOrder(), step.getDescription(), step.getClass().getSimpleName());
        }
        plan.addSteps(newArrayList(steps));
    }

    @Override
    public void addCheckpoint(Step step, Checkpoint checkpoint) {
        Delta delta = checkpoint.getDeltaToCheckpoint();
        Operation overrideOperation = checkpoint.getOverrideOperation();
        if (overrideOperation != null) {
            if (delta.getOperation() != Operation.MODIFY) {
                checkArgument(
                        delta.getOperation() == overrideOperation,
                        "The delta's operation (%s) and overrideOperation (%s) should match in non-modify scenarios", delta.getOperation(), overrideOperation);
            } else {
                checkArgument(overrideOperation != Operation.NOOP, "Cannot set a NOOP override operation on a MODIFY delta (%s)", delta);
            }
        }
        plan.getCheckpoints().add(new StepPlan.Checkpoint(delta, step, overrideOperation, checkpoint.getIntermediateCheckpointName()));
    }

    @Override
    public void addCheckpoint(Step step, Delta delta) {
        addCheckpoint(step, new Checkpoint(delta));
    }

    @Override
    public void addCheckpoint(final Step step, final Delta delta, final Operation overrideOperation) {
        addCheckpoint(step, new Checkpoint(delta, null, overrideOperation));
    }

    @Override
    public void addCheckpoint(Step step, Iterable<Delta> deltas) {
        for (Delta delta : deltas) {
            addCheckpoint(step, new Checkpoint(delta));
        }
    }

    @Override
    public void addStepWithCheckpoint(Step step, Checkpoint checkpoint) {
        addStep(step);
        addCheckpoint(step, checkpoint);
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Delta delta) {
        addStepWithCheckpoint(step, new Checkpoint(delta));
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Delta delta, final Operation overrideOperation) {
        addStepWithCheckpoint(step, new Checkpoint(delta, null, overrideOperation));
    }

    @Override
    public void addStepWithCheckpoint(final Step step, final Iterable<Delta> deltas) {
        addStep(step);
        for (Delta delta : deltas) {
            addCheckpoint(step, new Checkpoint(delta));
        }
    }

    @Override
    public Object getAttribute(String param) {
        return attributes.get(param);
    }

    @Override
    public void setAttribute(String param, Object value) {
        attributes.put(param, value);
        if (value instanceof ExecutionStateListener) {
            plan.getListeners().add((ExecutionStateListener) value);
        }
    }

    @Override
    public DeployedApplication getDeployedApplication() {
        return deployedApplication;
    }

    @Override
    public DeployedApplication getPreviousDeployedApplication() {
        return previousDeployedApplication;
    }

    @Override
    public Repository getRepository() {
        return readOnlyRepository;
    }

    @Override
    public String patch(String content, ConfigurationItem ci) {
        InputStream inputStream = patch(new ByteArrayInputStream(content.getBytes()), ci);
        try {
            return IOUtils.toString(inputStream, Charset.defaultCharset());
        } catch (IOException e) {
            throw new RuntimeException("IO Error while processing patch", e);
        }
    }

    @Override
    public InputStream patch(InputStream content, ConfigurationItem ci) {
        Map<String, Object> transformerContext = new HashMap<String, Object>(){{
            put("configurationItem", ci);
            put("deployed", Optional.ofNullable(deployedApplication).orElse(previousDeployedApplication));
        }};
        return getArtifactTransformer(PATCH_TRANSFORMER_NAME).transform(content, transformerContext);
    }

    @Override
    public ArtifactTransformer getArtifactTransformer(String transformerName) {
        ArtifactTransformer delegate = artifactTransformerFactory.getArtifactTransformer(transformerName);
        Map<String, Object> defaultContext = new HashMap<String, Object>(){{
            put(DEPLOYMENT_CONTEXT_KEY, new HashMap<String, Object>() {{
                put(DC_DEPLOYED_APPLICATION_KEY, deployedApplication);
                put(DC_PREVIOUS_DEPLOYED_APPLICATION_KEY, previousDeployedApplication);
                put(DC_PLAN_KEY, plan);
                put(DC_ROLLBACK_FLAG_KEY, rollback);
            }});
        }};
        return new ContextInjectingArtifactTransformer(delegate, defaultContext);
    }

    public boolean isRollback() {
        return rollback;
    }

    private static final Logger logger = LoggerFactory.getLogger(DefaultDeploymentPlanningContext.class);
}
