package com.xebialabs.deployit.inspection;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.Step.Result;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.inspection.InspectionStep;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Discovers ConfigurationItems from reality and/or syncs them.
 */
public class Inspector {

    public static void inspect(ConfigurationItem item, com.xebialabs.deployit.plugin.api.inspection.InspectionContext context) {
        checkRequiredDiscoveryProperties(item);
        Method inspectMethod = findInspectMethod(item);

        if (inspectMethod != null) {
            logger.debug("Invoking @Inspect method on {}", item);
            Class<?>[] parameterTypes = inspectMethod.getParameterTypes();

            if (parameterTypes[0].equals(InspectionPlanningContext.class)) {
                callInspectionMethod(inspectMethod, item, new InspectionContextAdapter(context));
            } else if (parameterTypes[0].equals(com.xebialabs.deployit.plugin.api.inspection.InspectionContext.class)) {
                callInspectionMethod(inspectMethod, item, context);
            }
            // DEPLOYITPB-3766 Enabled here to revert to previous situation (see below)
            context.inspected(item);
        }

        // Always call inspected. If there is no inspectMethod, this CI needs no further actions thus it is inspected.
        // DEPLOYITPB-3766 Disabled because it breaks WAS itests
//        context.inspected(item);
    }

    private static void callInspectionMethod(Method inspectMethod, ConfigurationItem item, Object ctx) {
        try {
            inspectMethod.invoke(item, ctx);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException("Cannot invoke @Inspect method on " + item, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Cannot invoke @Inspect method on " + item, e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException("Error invoking @Inspect method on " + item, e);
        }
    }

    private static void checkRequiredDiscoveryProperties(ConfigurationItem item) {
        Descriptor descriptor = item.getType().getDescriptor();
        for (PropertyDescriptor propertyDescriptor : descriptor.getPropertyDescriptors()) {
            if (propertyDescriptor.isInspectionProperty() && propertyDescriptor.isRequiredForInspection()) {
                Object o = propertyDescriptor.get(item);
                if (o == null) {
                    throw new IllegalArgumentException("Missing required property for discovery " + propertyDescriptor.getName());
                }
            }
        }
    }

    private static Method findInspectMethod(ConfigurationItem item) {
        Method[] methods = item.getClass().getMethods();
        for (Method method : methods) {
            if (method.isAnnotationPresent(Inspect.class)) {
                return method;
            }
        }
        return null;
    }

    private static final Logger logger = LoggerFactory.getLogger(Inspector.class);
}
