package com.xebialabs.deployit.deployment
package rules

import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, StepMetadata, StepPostConstructContext}
import com.xebialabs.platform.script.jython.JythonContext
import grizzled.slf4j.Logging

import java.lang.reflect.{Field, InvocationTargetException}
import java.util
import scala.jdk.CollectionConverters._

class StepFactory(stepRegistry: StepRegistry,
                  stepPostConstructContext: StepPostConstructContext,
                  xmlParamResolver: XmlParameterResolver) extends Logging with ContextFactory {

  def step(stepData: StepData, existingBindings: Map[String, Any] = Map()): Step = {
    val stepName: String = stepData.stepName
    val stepParameters: Map[String, Any] = stepData.stepParameters
    trace(s"Creating step $stepName with parameters $stepParameters")
    createStep(stepName, stepParameters, existingBindings)
  }

  def step(stepName: String, stepParams: Seq[XmlParameter], existingBindings: Map[String, Any]): Step = {
    implicit val jc: JythonContext = jythonContext(existingBindings)
    val resolvedUserParams: Map[String, Any] = xmlParamResolver.resolveXmlParameters(stepName, stepParams)
    createStep(stepName, resolvedUserParams, existingBindings)
  }

  private[this] def createStep(stepName: String, resolvedParams: Map[String, Any], existingBindings: Map[String, Any] = Map()): Step = {
    stepRegistry.getStepDescriptor(stepName) match {
      case sd: StepMacroDescriptor =>
        step(StepData(sd.stepData.stepName, resolveParamsAgainstMacroDict(resolvedParams, existingBindings, sd)))
      case sd: StepPrimitiveDescriptor =>
        createStepByDescriptor(sd, resolvedParams)
    }
  }

  private[this] def resolveParamsAgainstMacroDict(resolvedParams: Map[String, Any], existingBindings: Map[String, Any], sd: StepMacroDescriptor): Map[String, Any] = {
    implicit val jc: JythonContext = jythonContext(existingBindings ++ macroBinding(resolvedParams.asJava))
    xmlParamResolver.resolveXmlParameters(sd.stepData.stepName, sd.stepData.stepParameters)
  }

  private[this] def macroBinding(macroDictionary: util.Map[String, Any]): Map[String, Any] = Map[String, Any]("macro" -> macroDictionary)

  private def createStepByDescriptor(stepDescriptor: StepPrimitiveDescriptor, stepParameterValues: Map[String, Any]): Step = {
    val step = stepDescriptor.implementationClass.newInstance()
    stepParameterValues.foreach { case (name, value) =>
      val parameter = stepDescriptor.parameters.getOrElse(name, throw new IllegalArgumentException(s"Parameter '$name' could not be found for step ${stepDescriptor.name}"))
      setStepParameterValue(step, parameter.asInstanceOf[StepPrimitiveParameterDescriptor], value)
    }
    doPostConstruction(step)
    validateAllRequiredParametersAreSet(stepDescriptor, step)
    step
  }

  private def setStepParameterValue(step: Step, descriptor: StepPrimitiveParameterDescriptor, value: Any): Unit = {
    val parameterName: String = descriptor.name
    val field = descriptor.field
    val javaHappyValue = descriptor.parameterType match {
      case MapParameterType if value.isInstanceOf[Map[_, _]] => new util.HashMap[String, Any](value.asInstanceOf[Map[String, Any]].asJava)
      case ListParameterType if value.isInstanceOf[List[_]] => new util.ArrayList[Any](value.asInstanceOf[List[Any]].asJava)
      case _ => value
    }
    field.setAccessible(true)
    try {
      field.set(step, javaHappyValue)
    } catch {
      case _: Exception => throw new IllegalArgumentException(s"Value of type ${Option(javaHappyValue).map(_.getClass.getName).getOrElse("[null]")} cannot be assigned to parameter '$parameterName' of type ${field.getType.getName}")
    }
  }

  private def validateAllRequiredParametersAreSet(stepDescriptor: StepPrimitiveDescriptor, step: Step): Unit = {
    def notNullAndBlank(value: Any) = value != null && value.toString != ""

    def getValue(anyRef: AnyRef, field: Field) = {
      field.setAccessible(true)
      field.get(anyRef)
    }

    stepDescriptor.parameters.filter(_._2.required).foreach { case (paramName, paramDescriptor) =>
      require(notNullAndBlank(getValue(step, paramDescriptor.field)),
        s"Cannot create step '${stepDescriptor.name}' since required parameter '$paramName' is not specified")
    }
  }

  private def doPostConstruction(step: Step): Unit = {
    trace(s"post-constructing [${step.getClass.getName}]")
    getClassChain(step.getClass)
      .flatMap(_.getDeclaredMethods)
      .filter(m => m.getAnnotation(classOf[RulePostConstruct]) != null)
      .sortBy(_.getName)
      .foreach(m => {
        debug(s"${step.getClass.getSimpleName} invoking [${m.getName}]")
        m.setAccessible(true)
        try {
          m.invoke(step, stepPostConstructContext)
        } catch {
          case e: InvocationTargetException =>
            throw PlannerException(s"Error while post-constructing [${step.getClass.getAnnotation(classOf[StepMetadata]).name()}] step: ${e.getTargetException.getMessage}", e)
        }
      })
    trace(s"done post-constructing [${step.getClass.getName}]")
  }
}

case class StepData(stepName: String, stepParameters: Map[String, Any] = Map.empty)
