package com.xebialabs.deployit.deployment.planner;

import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecificationWithDependencies;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toList;

public class MultiDeltaSpecification implements DeltaSpecificationWithDependencies {
    private final List<List<DeltaSpecification>> deltaSpecifications;
    private final DeployedApplication mainApplication;

    public MultiDeltaSpecification(final DeltaSpecification deltaSpecification) {
        this(deltaSpecification, Collections.emptyList());
    }

    private MultiDeltaSpecification(final DeltaSpecification mainDeltaSpecification, final List<List<DeltaSpecification>> dependencies) {
        checkNotNull(mainDeltaSpecification, "Wrong usage of internal API: mainDeltaSpecification must not be null");
        this.deltaSpecifications = dependencies.stream().map(ArrayList::new).collect(toList());
        List<DeltaSpecification> emptyList = new ArrayList<>();
        emptyList.add(mainDeltaSpecification);
        this.deltaSpecifications.add(emptyList);
        this.mainApplication = mainDeltaSpecification.getDeployedApplication();
    }

    private MultiDeltaSpecification(final List<List<DeltaSpecification>> allDeltaSpecifications, DeployedApplication mainApplication) {
        checkArgument(!allDeltaSpecifications.isEmpty(), "Wrong usage of internal API: allDeltaSpecifications must contain 1 element");
        this.deltaSpecifications = allDeltaSpecifications;
        this.mainApplication = mainApplication;
    }

    private Stream<List<DeltaSpecification>> deltaSpecificationStream() {
        return deltaSpecifications.stream();
    }

    @Override
    public List<DeltaSpecification> getAllDeltaSpecifications() {
        return flattenDeltaSpecs();
    }

    private List<DeltaSpecification> flattenDeltaSpecs() {
        List<DeltaSpecification> deltaSpecs = new ArrayList<>();
        deltaSpecificationStream().forEach(deltaSpecs::addAll);
        return deltaSpecs;
    }

    @Override
    public List<List<DeltaSpecification>> getAllGroupedDeltaSpecifications() {
        return deltaSpecifications;
    }

    public List<DeployedApplication> getAllDeployedApplications() {
        return flattenDeltaSpecs().stream().map(MultiDeltaSpecificationHelper::correctDeployedApplication).collect(toList());
    }

    public List<Delta> getAllDeltas() {
        return flattenDeltaSpecs().stream().flatMap(spec -> spec.getDeltas().stream()).collect(toList());
    }

    public List<String> getOrchestrators() {
        return Collections.unmodifiableList(mainApplication.getOrchestrator());
    }

    public DeployedApplication getMainApplication() {
        return mainApplication;
    }

    public static MultiDeltaSpecification withDependencies(DeltaSpecification main, List<List<DeltaSpecification>> dependencies) {
        return new MultiDeltaSpecification(main, dependencies);
    }

    public static MultiDeltaSpecification forRollback(List<List<DeltaSpecification>> deltaSpecifications, DeployedApplication mainApplication) {
        return new MultiDeltaSpecification(deltaSpecifications, mainApplication);
    }

    public static MultiDeltaSpecification forUndeploy(List<List<DeltaSpecification>> deltaSpecifications, DeployedApplication mainApplication) {
        return new MultiDeltaSpecification(deltaSpecifications, mainApplication);
    }
}
