package ai.digital.engine.exportcis

import java.io.{FileNotFoundException, InputStream}
import java.util.zip.ZipFile

import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.xlplatform.utils.ManagedResource
import grizzled.slf4j.Logging
import org.jdom2.{Element, Namespace}

import scala.io.Source._
import scala.language.implicitConversions
import scala.util.{Failure, Try}

package object archive extends Logging {

  implicit class ReadFriendlyZip(zip: ZipFile) {
    def readTextEntry(path: String): String = {
      val entry = zip.getEntry(path)
      fromInputStream(zip.getInputStream(entry)).mkString
    }

    def entryStream(path: String): InputStream = zip.getEntry(path) match {
      case null => throw new FileNotFoundException(s"Expected to find [$path] in ZIP file ${zip.getName}")
      case e => zip.getInputStream(e)
    }
  }

  implicit class ScalaFriendlyElement(e: Element)(implicit ns: Namespace) {

    def child(path: String*): Element = childOption(path:_*).getOrElse {
      throw new IllegalStateException(s"Can not find a mandatory element ${path.mkString("/")}")
    }

    def childOption(path: String*): Option[Element] = path.toList match {
      case Nil => throw new IllegalArgumentException(s"At least one path element is required")
      case x :: Nil => Option(e.getChild(x, ns))
      case x :: xs => Option(e.getChild(x, ns)).flatMap(_.childOption(xs:_*))
    }
  }

  implicit def workDir2managed(workDir: WorkDir): ManagedResource[WorkDir] = {
    new ManagedResource[WorkDir] {
      override def onEnter(): WorkDir = workDir
      override def onExit(exceptionHappened: Boolean): Unit = {
        Try(workDir.delete()) match {
          case Failure(e) => logger.warn(s"Could not delete work directory $workDir, you can do this manually")
          case _ =>
        }
      }
    }
  }
}
