package ai.digital.engine.exportcis.archive

import java.io._
import java.nio.file.Path
import java.util.zip.{ZipEntry, ZipOutputStream}

import ai.digital.engine.exportcis.archive.RepositoryExporter._
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.repository.RepositoryService
import com.xebialabs.overthere.util.OverthereUtils
import com.xebialabs.xlplatform.sugar.PathSugar._
import com.xebialabs.xlplatform.utils.ResourceManagement._
import org.apache.commons.lang.CharEncoding

import scala.util.Try

object RepositoryExporter {
  val CiXmlFileName = "configuration-items.xml"
  val RepositoryRootAlias = "/"
}

class RepositoryExporter(val repositoryService: RepositoryService, val ctx: ExecutionContext) {

  private val xmlExporter = new RepositoryXmlExporter(repositoryService, ctx)

  def export(ciRoots: Iterable[String], exportFile: Path, userName: String, appVersion: String): Try[Unit] = {

    Try {
      val result: RepositoryXmlExporterResult = xmlExporter.exportCiTrees(ciRoots, userName, appVersion)

      ctx.logOutput(s"Started packing CIs metadata and contents into a ZIP file: $exportFile")

      using(new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(exportFile.jfile())))) { zos =>
        zos.putNextEntry(new ZipEntry(CiXmlFileName))
        zos.write(result.xmlContent.getBytes(CharEncoding.UTF_8))
        zos.closeEntry()
        for ((path, overthereContent) <- result.filesByZipEntry) {

          zos.putNextEntry(new ZipEntry(path))

          if (overthereContent.isDirectory) {
            writeFolderAsZip(new File(overthereContent.getPath), zos)
          } else {
            using(overthereContent.getInputStream) { fileStream =>
              OverthereUtils.write(fileStream, zos)
            }
          }

          zos.closeEntry()
        }

        ctx.logOutput(s"Finished packing CIs metadata and contents into a ZIP file: $exportFile")
      }
    }
  }

  private[archive] def writeFolderAsZip(folder: File, os: OutputStream): Unit = {

    def writeIntoZip(file: File, zos: ZipOutputStream, basePath: String = ""): Unit = {
      if (!file.isDirectory) {
        zos.putNextEntry(new ZipEntry(s"$basePath${file.getName}"))
        using(new FileInputStream(file)) { fis =>
          OverthereUtils.write(fis, zos)
        }
        zos.closeEntry()
      } else {
        for (child <- file.listFiles()) {
          writeIntoZip(child, zos, s"$basePath${file.getName}/")
        }
      }
    }

    val folderAsZipStream = new ZipOutputStream(os)
    for (file <- folder.listFiles()) {
      writeIntoZip(file, folderAsZipStream)
    }
    // Avoid closing the parent output stream
    folderAsZipStream.finish()
  }

}
