package ai.digital.deploy.cache.actuator

import ai.digital.configuration.central.deploy.CachesProperties
import ai.digital.deploy.cache.config.{CacheMode, CacheProvider}
import ai.digital.deploy.cache.service.{AppCacheDataService, CacheDataService}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.actuate.endpoint.annotation.{DeleteOperation, Endpoint, ReadOperation, Selector}
import org.springframework.boot.actuate.endpoint.web.WebEndpointResponse
import org.springframework.context.annotation.DependsOn
import org.springframework.stereotype.Component

import scala.util.{Failure, Success, Try}
import java.lang.reflect.ParameterizedType
import java.util
import java.util.{List => JList, Map => JMap}
import scala.jdk.CollectionConverters._

@Component
@DependsOn(value = Array("cacheMode"))
@Endpoint(id = "jcaches")
class JCachesEndpoint(@Autowired cachesProperties: CachesProperties,
                      @Autowired(required = false) cacheProvider: CacheProvider,
                      @Autowired cacheDataServices: JList[AppCacheDataService[_ <: AnyRef, _ <: AnyRef]])
  extends Logging {

  @ReadOperation
  def info(): CachesInfo = {
    CachesInfo(
      Configured(cachesProperties.appCaches.enabled),
      Evaluated(
        CacheMode.getCacheMode,
        if (cacheProvider != null) cacheProvider.getType else null,
        cacheProvider != null
      ),
      getCaches
    )
  }

  @ReadOperation
  def cacheEntry(@Selector cacheName: String, key: String): WebEndpointResponse[_ <: AnyRef] = {
    val notFoundResponse = new WebEndpointResponse(null, WebEndpointResponse.STATUS_NOT_FOUND)
    withCacheProviderCheck({
      getCacheDataService(cacheName) match {
        case Some(cds) =>
          val keyType = getCacheDataServiceKeyType(cds)
          getCacheValue(key, keyType, cds)
        case None => notFoundResponse
      }
    })
  }

  @DeleteOperation
  def clearCaches(): WebEndpointResponse[Unit] = {
    withCacheProviderCheck({
      if (cacheDataServices != null && !cacheDataServices.isEmpty) {
        cacheDataServices.forEach(
          cacheDataService => cacheDataService.clear()
        )
        new WebEndpointResponse(WebEndpointResponse.STATUS_OK)
      } else {
        new WebEndpointResponse(WebEndpointResponse.STATUS_BAD_REQUEST)
      }
    })
  }

  @DeleteOperation
  def clearCache(@Selector cacheName: String): WebEndpointResponse[Unit] = {
    withCacheProviderCheck({
      if (cacheDataServices != null && !cacheDataServices.isEmpty) {
        cacheDataServices.asScala
          .find(_.cacheName.equals(cacheName)) match {
          case Some(cds) =>
            cds.clear()
            new WebEndpointResponse(WebEndpointResponse.STATUS_OK)
          case None => new WebEndpointResponse(WebEndpointResponse.STATUS_NOT_FOUND)
        }
      } else {
        new WebEndpointResponse(WebEndpointResponse.STATUS_BAD_REQUEST)
      }
    })
  }

  private def withCacheProviderCheck[K](block: => WebEndpointResponse[K]): WebEndpointResponse[K] = {
    try {
      if (cacheProvider == null) {
        new WebEndpointResponse(WebEndpointResponse.STATUS_BAD_REQUEST)
      } else {
        block
      }
    } catch {
      case e: Exception =>
        logger.error(s"Exception occured. ${e.getMessage}")
        new WebEndpointResponse(WebEndpointResponse.STATUS_INTERNAL_SERVER_ERROR)
    }
}

  private def getCaches: JList[String] = {
    val caches: JList[String] = new util.ArrayList[String]()
    if (cacheProvider != null && cacheDataServices != null && !cacheDataServices.isEmpty) {
      cacheDataServices.forEach(cacheDataService => {
        if(cacheDataService.cacheConfiguration.enabled)
          caches.add(cacheDataService.cacheName)
      })
    }
    caches
  }

  private def getCacheDataService(cacheName: String): Option[CacheDataService[_ <: AnyRef, _ <: AnyRef]] = {
    if (cacheDataServices != null && !cacheDataServices.isEmpty) {
      cacheDataServices.asScala.find(_.cacheName.equals(cacheName))
    } else None
  }

  private def getCacheDataServiceKeyType(cds: CacheDataService[_ <: AnyRef, _ <: AnyRef]): Class[_] = {
    val genericTypes = cds.getClass.getGenericInterfaces
    val genericType = genericTypes(0).asInstanceOf[ParameterizedType].getActualTypeArguments
    genericType(0).asInstanceOf[Class[_]]
  }

  private def getCacheValue(key: String,
                          target: Class[_],
                          cds: CacheDataService[_ <: AnyRef, _ <: AnyRef]): WebEndpointResponse[_ <: AnyRef] = {
    val notFoundResponse = new WebEndpointResponse(null, WebEndpointResponse.STATUS_NOT_FOUND)
    val result = target match {
      case _ if target == classOf[Integer] =>
        Try(Integer.valueOf(key)) match {
          case Success(value) => cds.asInstanceOf[CacheDataService[Integer, _ <: AnyRef]].get(value)
          case Failure(_: NumberFormatException) => None
        }
      case _ if target == classOf[String] =>
        cds.asInstanceOf[CacheDataService[String, _ <: AnyRef]].get(key)
      case _ =>
        throw new Exception("Unknown type for key")
    }
    if (result.isDefined) {
      new WebEndpointResponse(result.get, WebEndpointResponse.STATUS_OK)
    } else notFoundResponse
  }

}

case class CachesInfo(configured: Configured,
                      evaluated: Evaluated,
                      caches: JList[String])

case class CacheEntriesReport(size: Int,
                              entries: JMap[_ <: AnyRef, _ <: AnyRef])

case class Configured(cacheEnabled: Boolean)

case class Evaluated(cacheMode: String,
                     provider: String,
                     providerAvailable: Boolean)
