package ai.digital.deploy.cache.config

import ai.digital.configuration.central.deploy.CachesProperties
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import javax.cache.Caching
import javax.cache.spi.CachingProvider
import scala.jdk.CollectionConverters._

@Component
class CacheConfigurationValidator(@Autowired cachesProperties: CachesProperties)
  extends Logging {

  val providers: Seq[CachingProvider] = Caching.getCachingProviders.asScala.toList

  def validate: Boolean = {
    val predicates = List(
      () => validateEmbeddedProvider,
      () => validateProviderConfigurationFile,
      () => validateProviderForExternal,
      () => validateProviderClassWithMultipleProviders
    )
    predicates.forall(p => p())
  }

  private def validateEmbeddedProvider: Boolean = {
    if (CacheMode.embeddedCacheMode) {
      if (providers.nonEmpty && providers.exists(
        p => p.toString.contains(DEFAULT_EMBEDDED_CACHING_PROVIDER))) {
        true
      } else {
        logger.error(s"Default embedded caching provider($DEFAULT_EMBEDDED_CACHING_PROVIDER) " +
          s"not found in classpath")
        false
      }
    } else {
      true
    }
  }

  private def validateProviderForExternal: Boolean = {
    if (CacheMode.embeddedCacheMode ||
      (!CacheMode.embeddedCacheMode &&
        providers.size > 1)) {
      true
    } else {
      logger.error(s"Provider not found for standalone cache server. " +
        s"Add the libraries for standalone cache server in the 'lib' directory")
      false
    }
  }

  private def validateProviderClassWithMultipleProviders: Boolean = {
    if (CacheMode.embeddedCacheMode) {
      true
    } else {
      if (providers.size > 2) {
        if (StringUtils.hasText(cachesProperties.providerConfiguration.provider)) {
          true
        } else {
          logger.error(s"Multiple cache providers (${providers.map(p => p.getClass.getName)}) detected in the classpath. " +
            s"Specify a provider class in the 'deploy.caches.provider-configuration.provider' field")
          false
        }
      } else {
        true
      }
    }
  }

  private def validateProviderConfigurationFile: Boolean = {
    if (CacheMode.embeddedCacheMode) {
      true
    } else if (!CacheMode.embeddedCacheMode &&
        StringUtils.hasText(cachesProperties.providerConfiguration.providerConfigurationFile)) {
      val fileResource = getResource(cachesProperties.providerConfiguration.providerConfigurationFile)
      if (fileResource == null) {
        logger.error(s"File not found in provider-configuration-file mentioned path")
        false
      } else {
        true
      }
    } else {
      logger.error(s"For standalone cache server, provider-configuration-file should be specified and " +
        s"should be resolved to a valid file on the classpath")
      false
    }
  }
}
