package ai.digital.deploy.cache.config

import ai.digital.configuration.central.deploy.CacheConfiguration
import ai.digital.deploy.cache.service.CacheWrapper
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import java.util.concurrent.TimeUnit
import javax.cache.CacheManager
import javax.cache.configuration.{FactoryBuilder, MutableConfiguration}
import javax.cache.expiry.{AccessedExpiryPolicy, Duration}

@Component
class CacheProvider(private val cacheManager: CacheManager) extends Logging {

  def registerCache[K,V](cacheName: String, cacheConfiguration: CacheConfiguration): CacheWrapper[K,V] = {
    try {
      if (cacheManager.getCache[K,V](cacheName) == null) {
        val config = getConfiguration[K,V](cacheConfiguration)
        cacheManager.createCache[K,V, MutableConfiguration[K,V]](cacheName, config)
      }
      new CacheWrapper(cacheManager.getCache[K,V](cacheName))
    } catch {
      case exception: Exception =>
        logger.error(s"Exception occurred while creating cache($cacheName), $exception")
        null
    }
  }

  def getType: String = {
    cacheManager.getCachingProvider.toString
  }

  def close(): Unit = {
    logger.info(s"Closing the cache manager")
    cacheManager.close()
  }

  private def getConfiguration[K,V](cacheConfiguration: CacheConfiguration): MutableConfiguration[K,V] = {
    val config = new MutableConfiguration[K,V]()
    config.setExpiryPolicyFactory(FactoryBuilder.factoryOf(
      new AccessedExpiryPolicy(
        new Duration(TimeUnit.MINUTES, cacheConfiguration.ttlMinutes))))
    config.setManagementEnabled(cacheConfiguration.managementEnabled)
    config.setStatisticsEnabled(cacheConfiguration.statisticsEnabled)
  }
}
