package ai.digital.configuration.central.deploy

import com.typesafe.config.ConfigFactory
import grizzled.slf4j.Logging
import org.springframework.util.StreamUtils

import java.io.File
import java.nio.file.{Files, OpenOption, Path, Paths}
import scala.util.Using

object InitialCentralConfigSetup extends Logging {

  private case class DiffResult(
                                 filePath: Path,
                                 generator: Option[() => Unit] = None
                               )

  private val CENTRAL_CONFIG_PATH = "centralConfiguration"
  private val CONFIG_FOLDER = new File("conf")
  private val REPOSITORY_YAML_NAME = "deploy-repository.yaml"
  private val XL_DEPLOY_CONF_NAME = "xl-deploy.conf"


  private val initialCentralConfFiles = Set(
    "deploy-artifact-resolver.yaml",
    "deploy-client.yaml",
    "deploy-cluster.yaml",
    "deploy-command-whitelist.yaml",
    "deploy-db-anonymizer.yaml",
    "deploy-jmx.yaml",
    "deploy-metrics.yaml",
    "deploy-oidc.yaml",
    "deploy-plugins.yaml",
    REPOSITORY_YAML_NAME,
    "deploy-satellite.yaml",
    "deploy-secret-complexity.yaml",
    "deploy-server.yaml",
    "deploy-task.yaml",
    "deploy-websockets.yaml",
    "deploy-caches.yaml"
  )

  private lazy val xlDeployConf = if (new File(CONFIG_FOLDER, XL_DEPLOY_CONF_NAME).exists())
    ConfigFactory.load(XL_DEPLOY_CONF_NAME)
  else
    ConfigFactory.defaultReference()

  private val checkMainRepositoryPropertiesNotExist: Path => Boolean = path =>
    !path.endsWith(REPOSITORY_YAML_NAME) ||
      (path.endsWith(REPOSITORY_YAML_NAME) && !xlDeployConf.hasPath("xl.repository.database.db-url"))

  def setup(): Set[Path] = {
    createCentralConfigDir()
    val initialDiffs = diffInitialFiles()

    if (initialDiffs.nonEmpty) {
      logger.info("Copying central configuration files if necessary.")
      copyCentralConfigConfToFilesystemBackend(initialDiffs)
    } else {
      Set.empty
    }
  }

  protected def copyCentralConfigConfToFilesystemBackend(diffs: Set[DiffResult]): Set[Path] = {
    val classLoader = Thread.currentThread().getContextClassLoader

    diffs.map(diff => {
      Using.resource(classLoader.getResourceAsStream(diff.filePath.toString.replace("\\", "/"))) { fileStream =>
        val bytes = StreamUtils.copyToByteArray(fileStream)
        writeToFile(diff.filePath, bytes)
      }
    })
  }

  private def writeToFile(path: Path, content: Array[Byte], options: OpenOption*): Path = {
    Using.resource(Files.newOutputStream(path, options: _*)) { writer =>
      writer.write(content)
    }
    path
  }

  private def diffInitialFiles(): Set[DiffResult] = {
    val existingFiles = initialCentralConfFiles
      .map(Paths.get(CENTRAL_CONFIG_PATH, _))
      .filter(!Files.exists(_))
      .filter(checkMainRepositoryPropertiesNotExist)

    existingFiles
      .map(DiffResult(_))
  }

  private def createCentralConfigDir(): Unit = {
    val centralConfigDir = Paths.get(CENTRAL_CONFIG_PATH)
    if (!Files.exists(centralConfigDir))
      Files.createDirectories(centralConfigDir)
  }
}
