package ai.digital.configuration.central.deploy

import ai.digital.doc.properties.DocPropertiesFile
import com.fasterxml.jackson.annotation.JsonProperty
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

import java.beans.BeanProperty

/**
 * Enable and configure application level caching.
 * The caching is based on the standard JCache API(JSR 107).
 */

@Configuration
@ConfigurationProperties(prefix = "deploy.caches")
@DocPropertiesFile("deploy-caches.yaml")
class CachesProperties {

  /**
   * Cache App Caches configuration
   */
  @BeanProperty
  @JsonProperty("app-caches")
  var appCaches: AppCaches = new AppCaches

  /**
   * Cache System Caches configuration
   */
  @BeanProperty
  @JsonProperty("system-caches")
  var systemCaches: SystemCaches = new SystemCaches

  override def toString = s"CachesProperties(appCaches=$appCaches, systemCaches=$systemCaches)"
}

class AppCaches {
  /**
   * Toggle the overall caching at the application level
   */
  @BeanProperty
  var enabled: Boolean = false

  /**
   * Cache provider configuration
   */
  @BeanProperty
  @JsonProperty("provider-configuration")
  var providerConfiguration: ProviderConfiguration = new ProviderConfiguration

  /**
   * Circuit breaker Configuration
   */
  @BeanProperty
  @JsonProperty("circuit-breaker-configuration")
  var circuitBreakerConfiguration: CircuitBreakerConfiguration = new CircuitBreakerConfiguration

  /**
   * License cache configuration. Default ttl is 1440 minutes
   */
  @BeanProperty
  @JsonProperty("license-cache-configuration")
  var licenseCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 1440
  }

  /**
   * CI by pk cache configuration. Default ttl is 60 minutes
   */
  @BeanProperty
  @JsonProperty("ci-pk-cache-configuration")
  var ciPkCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 60
  }

  /**
   * CI by path cache configuration. Default ttl is 60 minutes
   */
  @BeanProperty
  @JsonProperty("ci-path-cache-configuration")
  var ciPathCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 60
  }

  /**
   * CI PROPERTIES cache configuration
   */
  @BeanProperty
  @JsonProperty("ci-properties-cache-configuration")
  var ciPropertiesCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 60
  }

  override def toString = s"AppCaches(enabled=$enabled, providerConfiguration=$providerConfiguration, licenseCacheConfiguration=$licenseCacheConfiguration, ciPkCacheConfiguration=$ciPkCacheConfiguration, ciPathCacheConfiguration=$ciPathCacheConfiguration,  ciPropertiesCacheConfiguration=$ciPropertiesCacheConfiguration)"

}

class SystemCaches {
  /**
   * Toggle the overall caching at the application level
   */
  @BeanProperty
  var enabled: Boolean = false

  /**
   * JYTHON SCRIPT CACHE PROPERTIES cache configuration
   */
  @BeanProperty
  @JsonProperty("jython-script-cache-configuration")
  var jythonScriptCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 5
  }

  override def toString = s"SystemCaches(enabled=$enabled,jythonScriptCacheConfiguration=$jythonScriptCacheConfiguration)"

}

class ProviderConfiguration {

  /**
   * Set the provider className to be passed to the Caching.getCachingProvider method.
   * An embedded Infinispan cache provider is used in single instance setup(single master with in-process worker) by default.
   * In all other setups a standalone cache server should be used and provider value should be configured.
   */
  @BeanProperty
  var provider: String = _

  /**
   * The provider specific configuration file name. This file should be on the classpath of all the
   * Deploy services(Master, Worker, Permission Service)
   */
  @BeanProperty
  @JsonProperty("provider-configuration-file")
  var providerConfigurationFile: String = _

  override def toString = s"ProviderConfiguration(provider=$provider, providerConfigurationFile=$providerConfigurationFile)"
}

class CircuitBreakerConfiguration {

  /**
   * Configures the failure rate threshold in percentage. If the failure rate is equal to or greater than the threshold,
   * the CircuitBreaker transitions to open and starts short-circuiting calls.
   * The threshold must be greater than 0 and not greater than 100. Default value is 50 percentage.
   */
  @BeanProperty
  @JsonProperty("failure-rate-threshold-percentage")
  var failureRateThresholdInPercentage: Int = 50

  /**
   * Configures an interval function with a fixed wait duration which controls how long the CircuitBreaker should stay open,
   * before it switches to half open. Default value is 120 seconds.
   */
  @BeanProperty
  @JsonProperty("wait-duration-in-open-state-seconds")
  var waitDurationInOpenStateInSeconds: Int = 120

  /**
   * Configures the minimum number of calls which are required (per sliding window period) before the CircuitBreaker
   * can calculate the error rate. For example, if minimumNumberOfCalls is 10, then at least 10 calls must be recorded,
   * before the failure rate can be calculated. If only 9 calls have been recorded,
   * the CircuitBreaker will not transition to open, even if all 9 calls have failed.
   * Default minimumNumberOfCalls is 2
   */
  @BeanProperty
  @JsonProperty("minimum-number-of-calls")
  var minimumNumberOfCalls: Int = 2

  /**
   * Configures the size of the sliding window which is used to record the outcome of calls when the
   * CircuitBreaker is closed. slidingWindowSize configures the size of the sliding window.
   * The slidingWindowSize must be greater than 0. Default slidingWindowSize is 2.
   */
  @BeanProperty
  @JsonProperty("sliding-window-size")
  var slidingWindowSize: Int = 2

  /**
   * Configures the number of permitted calls when the CircuitBreaker is half open.
   * The size must be greater than 0. Default size is 2.
   */
  @BeanProperty
  @JsonProperty("permitted-number-of-calls-in-half-open-state")
  var permittedNumberOfCallsInHalfOpenState: Int = 2

  override def toString = s"CircuitBreakerConfiguration(failureRateThresholdInPercentage=$failureRateThresholdInPercentage, waitDurationInOpenStateInSeconds=$waitDurationInOpenStateInSeconds, minimumNumberOfCalls=$minimumNumberOfCalls, slidingWindowSize=$slidingWindowSize, permittedNumberOfCallsInHalfOpenState=$permittedNumberOfCallsInHalfOpenState)"
}

class CacheConfiguration {
  /**
   * Toggle the individual cache
   */
  @BeanProperty
  var enabled: Boolean = true

  /**
   * Enable or disable management. If enabled the CacheMXBean for cache is registered
   * in the platform MBean server
   */
  @BeanProperty
  @JsonProperty("management-enabled")
  var managementEnabled: Boolean = true

  /**
   * Enable or disable statistics gathering for the cache
   */
  @BeanProperty
  @JsonProperty("statistics-enabled")
  var statisticsEnabled: Boolean = true

  /**
   * Time-to-live setting in minutes for entries in this cache
   */
  @BeanProperty
  @JsonProperty("ttl-minutes")
  var ttlMinutes: Int = 10

  override def toString = s"CacheConfiguration(enabled=$enabled, managementEnabled=$managementEnabled, statisticsEnabled=$statisticsEnabled, ttlMinutes=$ttlMinutes)"
}
