package ai.digital.configuration.central.deploy

import ai.digital.doc.properties.{DocDefaultValue, DocPropertiesFile}
import com.fasterxml.jackson.annotation.JsonProperty
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

import java.beans.BeanProperty
import java.util.{List => JList}

/**
  * Configuration to monitor the system via Kamon.
  */
@Configuration
@ConfigurationProperties(prefix = "deploy.jmx")
@DocPropertiesFile("deploy-jmx.yaml")
class JmxProperties {

  /**
    * Enabling the JMX.
    *
    */
  @BeanProperty
  @JsonProperty("enabled")
  var enabled: String = "no"

  /**
    * Host name for RMI server
    *
    */
  @BeanProperty
  @JsonProperty("hostname")
  var hostname: String = _

  /**
    * Port of the RMI server
    *
    */
  @BeanProperty
  @JsonProperty("port")
  var port: Int = 1099

  @BeanProperty
  @JsonProperty("ssl")
  var ssl: SSL = _

  /**
    * Internal configuration of Kamon.
    *
    * Check the tool itself for more information: https://github.com/kamon-io/kamon-jmx
    */
  @BeanProperty
  @JsonProperty("kamon")
  var kamon: Kamon = _

}

class SSL {
  /**
    * Should be set to true to enable SSL.
    *
    */
  @BeanProperty
  @JsonProperty("enabled")
  var enabled: String = "false"

  /**
    * Path to the key store.
    *
    */
  @BeanProperty
  @JsonProperty("key-store")
  var keyStore: String = _

  /**
    * Password for the key store.
    *
    */
  @BeanProperty
  @JsonProperty("key-store-password")
  var keyStorePassword: String = _

  /**
    * Path to a trust store.
    *
    */
  @BeanProperty
  @JsonProperty("trust-store")
  var trustStore: String = _

  /**
    * Password for a key store.
    *
    */
  @BeanProperty
  @JsonProperty("trust-store-password")
  var trustStorePassword: String = _

  /**
    * Protocol to use for SSL encryption.
    *
    * For example: 'TLSv1.1', 'TLSv1.2'.
    *
    */
  @BeanProperty
  @JsonProperty("protocol")
  var protocol: String = _

  /**
    * Example: ["TLS_RSA_WITH_AES_128_CBC_SHA", "TLS_RSA_WITH_AES_256_CBC_SHA"]
    * You need to install the JCE Unlimited Strength Jurisdiction Policy
    * Files to use AES 256.
    * More info here:
    * http://docs.oracle.com/javase/7/docs/technotes/guides/security/SunProviders.html#SunJCEP
    *
    */
  @BeanProperty
  @JsonProperty("enabled-algorithms")
  @DocDefaultValue(value="[]", removeQuote = true)
  var enabledAlgorithms: JList[String] = JList.of()

  /**
    * (Docs taken from https://doc.akka.io/docs/akka/2.6/general/configuration-reference.html)
    * There are two options, and the default SecureRandom is recommended:
    * "" or "SecureRandom" => (default)
    * "SHA1PRNG" => Can be slow because of blocking issues on Linux
    *
    * "AES128CounterSecureRNG" and "AES256CounterSecureRNG" were deprecated in Akka
    * 2.5.16 and if these configuration values are used it will use the default
    * SecureRandom anyway and log a warning. The deprecated implementations are
    * not recommended, but they can be enabled by using configuration values
    * "DeprecatedAES128CounterSecureRNG" or "DeprecatedAES256CounterSecureRNG"
    *
    * Setting a value here may require you to supply the appropriate cipher
    * suite (see enabled-algorithms section above)
    *
    */
  @BeanProperty
  @JsonProperty("random-number-generator")
  var randomNumberGenerator: String = _
}

class Kamon {

  /**
    * Controls whether the AspectJ Weaver missing warning should be displayed
    * if any Kamon module requiring AspectJ is found in the classpath
    * but the application is started without the AspectJ Weaver.
    */
  @BeanProperty
  @JsonProperty("show-aspectj-missing-warning")
  var showAspectjMissingWarning: String = "yes"

  @BeanProperty
  @JsonProperty("metric")
  var metric: Metric = _

  @BeanProperty
  @JsonProperty("kamon-mxbeans")
  var kamonMxbeans: KamonMxbeans = _

}

class Metric {

  /**
    * Setting decides whether to track or not the entities that do not match any 'includes' or 'excludes' filters.
    */
  @BeanProperty
  @JsonProperty("track-unmatched-entities")
  var trackUnmatchedEntities: String = _

  @BeanProperty
  @JsonProperty("filters")
  var filters: Filters = _
}

class Filters {

  /**
    * {{{
    * includes:
    *    - "PlannerActorSystem/\*\*"
    *    - "task-sys/\*\*"
    *    - "xl-scheduler-system/\*\*"
    * excludes: []
    * }}}
    */
  @BeanProperty
  @JsonProperty("pekko-dispatcher")
  var pekkoDispatcher: PekkoDispatcher = _
}

class PekkoDispatcher {

  /**
    * Gather metrics from dispatchers configured in 'includes' filter
    */
  @BeanProperty
  @JsonProperty("includes")
  @DocDefaultValue(value="[]", removeQuote = true)
  var includes: JList[String] = JList.of()

  /**
    * Exclude gathering metrics from dispatchers configured in 'excludes' filter
    */
  @BeanProperty
  @JsonProperty("excludes")
  @DocDefaultValue(value="[]", removeQuote = true)
  var excludes: JList[String] = JList.of()
}

class KamonMxbeans {

  /**
    * Configuring what mbean(s) that should be exported to Kamon.
    *
    */
  @BeanProperty
  @JsonProperty("mbeans")
  @DocDefaultValue(value="[]", removeQuote = true)
  var mbeans: JList[String] = JList.of()

  /**
    * The identify-delay-interval-ms is the number of milliseconds to wait before querying JMX the first time.
    *
    */
  @BeanProperty
  @JsonProperty("identify-delay-interval-ms")
  var identifyDelayIntervalMs: Int = 1000

  /**
    * The identify-interval-ms is how often to query JMX for new mxbeans.
    *
    */
  @BeanProperty
  @JsonProperty("identify-interval-ms")
  var identifyIntervalMs: Int = 1000

  /**
    * The value-check-interval-ms is the number of milliseconds between polling of known mxbeans for new metric values
    *
    */
  @BeanProperty
  @JsonProperty("value-check-interval-ms")
  var valueCheckIntervalMs: Int = 1000
}
