package ai.digital.configuration.central.deploy.db

import ai.digital.configuration.central.deploy.converter.HoconDurationConverter
import com.fasterxml.jackson.annotation.JsonProperty
import org.springframework.beans.factory.annotation.Value

import java.util.concurrent.TimeUnit
import scala.beans.BeanProperty

class DatabaseProperties {
  /**
    * This property controls the maximum number of milliseconds that a client (that's you) will wait for a connection from the pool.
    * If this time is exceeded without a connection becoming available, a SQLException will be thrown.
    * Lowest acceptable connection timeout is 250 ms.
    *
    */
  @BeanProperty
  @JsonProperty("connection-timeout")
  @Value("${connection-timeout}")
  var connectionTimeout: String = "30 seconds"

  /**
    * HikariCP will attempt to resolve a driver through the DriverManager based solely on the jdbcUrl,
    * but for some older drivers the driverClassName must also be specified.
    * Omit this property unless you get an obvious error message indicating that the driver was not found.
    *
    */
  @BeanProperty
  @JsonProperty("db-driver-classname")
  @Value("${db-driver-classname}")
  var dbDriverClassname: String = _

  /**
    * This property sets the default authentication password used when obtaining Connections from the underlying driver.
    * Note that for DataSources this works in a very deterministic fashion by calling DataSource.getConnection(username, *password*)
    * on the underlying DataSource. However, for Driver-based configurations, every driver is different.
    * In the case of Driver-based, HikariCP will use this password property to set a password property
    * in the Properties passed to the driver's DriverManager.getConnection(jdbcUrl, props) call.
    * If this is not what you need, skip this method entirely and call addDataSourceProperty("pass", ...), for example.
    *
    */
  @BeanProperty
  @JsonProperty("db-password")
  @Value("${db-password}")
  var dbPassword: String = _

  /**
    * This property directs HikariCP to use "DriverManager-based" configuration.
    * We feel that DataSource-based configuration (above) is superior for a variety of reasons (see below),
    * but for many deployments there is little significant difference.
    * When using this property with "old" drivers, you may also need to set the driverClassName property,
    * but try it first without. Note that if this property is used, you may still use DataSource properties
    * to configure your driver and is in fact recommended over driver parameters specified in the URL itself.
    *
    */
  @BeanProperty
  @JsonProperty("db-url")
  @Value("${db-url}")
  var dbUrl: String = _

  /**
    * This property sets the default authentication username used when obtaining Connections from the underlying driver.
    * Note that for DataSources this works in a very deterministic fashion by calling DataSource.getConnection(*username*, password)
    * on the underlying DataSource. However, for Driver-based configurations, every driver is different.
    * In the case of Driver-based, HikariCP will use this username property to set a user property in the Properties
    * passed to the driver's DriverManager.getConnection(jdbcUrl, props) call. If this is not what you need,
    * skip this method entirely and call addDataSourceProperty("username", ...), for example.
    *
    */
  @BeanProperty
  @JsonProperty("db-username")
  @Value("${db-username}")
  var dbUsername: String = _

  /**
    * This property controls the maximum amount of time that a connection is allowed to sit idle in the pool.
    * This setting only applies when minimumIdle is defined to be less than maximumPoolSize.
    * Idle connections will not be retired once the pool reaches minimumIdle connections.
    * Whether a connection is retired as idle or not is subject to a maximum variation of +30 seconds,
    * and average variation of +15 seconds. A connection will never be retired as idle before this timeout.
    * A value of 0 means that idle connections are never removed from the pool.
    * The minimum allowed value is 10 seconds.
    *
    */
  @BeanProperty
  @JsonProperty("idle-timeout")
  @Value("${idle-timeout}")
  var idleTimeout: String = "10 minutes"

  /**
    * This property controls the amount of time that a connection can be out of the pool before a message is
    * logged indicating a possible connection leak.
    * A value of 0 means leak detection is disabled. Lowest acceptable value for enabling leak detection is 2 seconds.
    *
    */
  @BeanProperty
  @JsonProperty("leak-detection-threshold")
  @Value("${leak-detection-threshold}")
  var leakDetectionThreshold: String = "2 minutes"

  /**
    * This property controls the maximum lifetime of a connection in the pool. An in-use connection will never be retired,
    * only when it is closed will it then be removed. On a connection-by-connection basis,
    * minor negative attenuation is applied to avoid mass-extinction in the pool.
    * We strongly recommend setting this value, and it should be several seconds shorter than any database
    * or infrastructure imposed connection time limit. A value of 0 indicates no maximum lifetime (infinite lifetime),
    * subject of course to the idleTimeout setting. The minimum allowed value is 30 minutes.
    *
    */
  @BeanProperty
  @JsonProperty("max-life-time")
  @Value("${max-life-time}")
  var maxLifeTime: String = "30 minutes"

  /**
    * This property controls the maximum size that the pool is allowed to reach, including both idle and in-use connections.
    * Basically this value will determine the maximum number of actual connections to the database backend.
    * A reasonable value for this is best determined by your execution environment.
    * When the pool reaches this size, and no idle connections are available, calls to getConnection() will
    * block for up to connectionTimeout milliseconds before timing out. Please read about pool sizing.
    *
    */
  @BeanProperty
  @JsonProperty("max-pool-size")
  @Value("${max-pool-size}")
  var maxPoolSize: Int = 10

  /**
    * This property controls the minimum number of idle connections that HikariCP tries to maintain in the pool.
    * If the idle connections dip below this value and total connections in the pool are less than maximumPoolSize,
    * HikariCP will make a best effort to add additional connections quickly and efficiently.
    * However, for maximum performance and responsiveness to spike demands, we recommend not setting this value
    * and instead allowing HikariCP to act as a fixed size connection pool.
    *
    */
  @BeanProperty
  @JsonProperty("minimum-idle")
  @Value("${minimum-idle}")
  var minimumIdle: Int = 10

  /**
    * This property sets the default schema for databases that support the concept of schemas.
    * If this property is not specified, the default schema defined by the JDBC driver is used.
    *
    */
  @BeanProperty
  @JsonProperty("db-schema-name")
  @Value("${db-schema-name}")
  var dbSchemaName: String = _

  def hasConfigured: Boolean = dbUrl != null && dbDriverClassname != null
  def getMaxLifeTimeInMillis: Long = HoconDurationConverter.convert(maxLifeTime, TimeUnit.MILLISECONDS)
  def getIdleTimeoutInMillis: Long = HoconDurationConverter.convert(idleTimeout, TimeUnit.MILLISECONDS)
  def getConnectionTimeoutInMillis: Long = HoconDurationConverter.convert(connectionTimeout, TimeUnit.MILLISECONDS)
  def getLeakDetectionThresholdInMillis: Long = HoconDurationConverter.convert(leakDetectionThreshold, TimeUnit.MILLISECONDS)
}
