package ai.digital.configuration.central.deploy

import ai.digital.configuration.central.deploy.server.HttpProperties
import ai.digital.doc.properties.{DocDefaultValue, DocPropertiesFile}
import com.fasterxml.jackson.annotation.JsonInclude.Include
import com.fasterxml.jackson.annotation.{JsonInclude, JsonProperty}
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.stereotype.Component

import java.util.{List => JList}
import scala.beans.BeanProperty

/**
  * Server specific configuration for a single deploy server master instance.
  */
@Component
@ConfigurationProperties(prefix = "deploy.server")
@JsonInclude(Include.NON_NULL)
@DocPropertiesFile("deploy-server.yaml")
class ServerSideProperties {

  /**
    * The host name to listen on.
    * The default setting will listen on all network interfaces.
    * A specific interface can be targeted by name or address.
    */
  @BeanProperty
  @JsonProperty("bind-hostname")
  var bindHostName: String = "0.0.0.0"

  /**
   * Server Http
   */
  @BeanProperty
  @JsonProperty
  var http = new HttpProperties

  /**
    * The label of the master instance.
    */
  @BeanProperty
  @JsonProperty
  var label: String = ""

  /**
    * Configuration of sending events
    */
  @BeanProperty
  @JsonProperty
  var events = new Events

  @BeanProperty
  @JsonProperty
  var ssl = new SslConfiguration

  @BeanProperty
  @JsonProperty
  var session = new ServerSideSession

  /**
    *  Timeout for gathering replies from workers
    */
  @BeanProperty
  @JsonProperty("aggregation-timeout")
  var aggregationTimeout = "5 seconds"

  @BeanProperty
  @JsonProperty("license")
  var license = new License

  /**
    * Where to cache download results for exported DARs and reports
    */
  @BeanProperty
  @JsonProperty
  var downloads = new Downloads

  @BeanProperty
  @JsonProperty
  var algorithms = new Algorithms

  @BeanProperty
  @JsonProperty("patch-dictionaries")
  var patchDictionaries = new PatchDictionaries

  /**
    * Enables the server to resolve application dependencies.
    *
    */
  @BeanProperty
  @JsonProperty("resolve-application-dependencies-enabled")
  var resolveApplicationDependenciesEnabled: Boolean = true

  @BeanProperty
  @JsonProperty("export-cis")
  var exportCis = new ExportCis

   /**
    * Enables to set maximum results size for pagination
    */
  @BeanProperty
  @JsonProperty("rest")
  var rest = new Rest

  /**
    * server extensions
    */
  @BeanProperty
  @JsonProperty("extensions")
  var extensions = new Extensions

  @BeanProperty
  @JsonProperty
  var security = new SecurityAuthProvider

  /**
   * server placeholders
   */
  @BeanProperty
  @JsonProperty("placeholders")
  var placeholders = new Placeholders

  /**
   * analytics
   */
  @BeanProperty
  @JsonProperty("analytics")
  var analytics = new Analytics

  /**
    * singleton scheduler
    */
  @BeanProperty
  @JsonProperty("singleton-scheduler")
  var singletonScheduler = new SingletonScheduler

  /**
    * Support Package Settings
    */

  @BeanProperty
  @JsonProperty("supportPackageSettings")
  var supportPackageSettings = new SupportPackageSettings

}

class SecurityAuthProvider {
  @BeanProperty
  @JsonProperty
  var auth = new {
    /**
     * To enable OIDC authentication set this token as "oidc"
     */
    @BeanProperty
    @JsonProperty
    @DocDefaultValue(value = "default", removeQuote = false)
    var provider: String = ServerSideProperties.DEFAULT_SECURITY_AUTH_PROVIDER
  }
}

class Rest {
  /**
   * rest.api
   */
  @BeanProperty
  @JsonProperty("api")
  var api = new Api
}

class Api {
  /**
   * maxPageSize - Pagination max results per page
   */
  @BeanProperty
  @JsonProperty("maxPageSize")
  @DocDefaultValue(value = "1000", removeQuote = true)
  var maxPageSize: Int = ServerSideProperties.DEFAULT_MAX_PAGE_SIZE
}

class Downloads {
  /**
    * The absolute folder name or relative to installation folder
    */
  @BeanProperty
  @JsonProperty("export-root")
  var exportRoot: String = "export"
}

@JsonInclude(Include.NON_EMPTY)
class License {
  /**
    * License expiry warning (in days).
    *
    */
  @BeanProperty
  @JsonProperty("days-before-warning")
  var daysBeforeWarning: Int = 5

  /**
    * Choice of 'byol' (Bring Your Own License), 'aws' (Container marketplace) or 'ami' (AMI marketplace)
    */
  @BeanProperty
  @JsonProperty
  var kind: String = "byol"

  /**
    * Where is the Digital.ai license endpoint for automated license fetches
    */
  @BeanProperty
  @JsonProperty
  var endpoint: String = "https://download.xebialabs.com/"

  @BeanProperty
  @JsonProperty
  var aws = new {
    /**
     * Product code
     */
    @BeanProperty
    @JsonProperty("product-code")
    var productCode: String = _

    /**
     * Version of AWS public key
     */
    @BeanProperty
    @JsonProperty("public-key-version")
    var publicKeyVersion: String = _

    /**
     * Aws public key
     */
    @BeanProperty
    @JsonProperty("public-key")
    var publicKey: String = _
  }

  @BeanProperty
  @JsonProperty("ami")
  var ami = new {
    /**
     * Product code
     */
    @BeanProperty
    @JsonProperty("product-code")
    var productCode: String = _
  }
}

class ServerSideSession {
  /**
   * Configures to clean up expired user sessions with the specified cron.
   *
   */
  @BeanProperty
  @JsonProperty("cleanup-cron")
  var cleanupCron: String = "0 0/20 * * * *"

  /**
   * To disable the “Active Sessions” screen, set this property to false. Once turned off the feature also stops all associated data collection.
   * If you are using MSSQL, we recommend that you disable “Active Sessions” to prevent deadlocks in the tables.
   *
   */
  @BeanProperty
  @JsonProperty("active-user-sessions-enabled")
  var activeUserSessionsEnabled: Boolean = true

  /**
   * Controls the maximum number of sessions for a user.
   * -1 to allow unlimited logins, or a positive integer to specify a maximum
   *
   */
  @BeanProperty
  @JsonProperty("maximumSessions")
  var maximumSessions: Int = -1

  /**
   * Determines the behaviour when a maximum sessions exceeded.
   * false to forcibly logged out the user that has already authenticated. true to throw an exception and prevent user to login
   *
   */
  @BeanProperty
  @JsonProperty("exceptionIfMaximumExceeded")
  var exceptionIfMaximumExceeded: Boolean = false
}

@JsonInclude(Include.NON_EMPTY)
class SslConfiguration {

  /**
    * Should be set to true to enable SSL.
    *
    */
  @BeanProperty
  var enabled: Boolean = false

  /**
    * Example: ["TLS_RSA_WITH_AES_128_CBC_SHA", "TLS_RSA_WITH_AES_256_CBC_SHA"]
    * You need to install the JCE Unlimited Strength Jurisdiction Policy
    * Files to use AES 256.
    * More info here:
    * http://docs.oracle.com/javase/7/docs/technotes/guides/security/SunProviders.html#SunJCEP
    *
    */
  @BeanProperty
  @JsonProperty("enabled-algorithms")
  @DocDefaultValue(value="[]", removeQuote = true)
  var enabledAlgorithms: JList[String] = JList.of()

  /**
    * Path to the key store.
    *
    */
  @BeanProperty
  @JsonProperty("key-store")
  var keyStore: String = "keystore"

  /**
    * Password for the key.
    *
    */
  @BeanProperty
  @JsonProperty("key-password")
  var keyPassword: String = "changeme"

  /**
    * Password for the key store.
    *
    */
  @BeanProperty
  @JsonProperty("key-store-password")
  var keyStorePassword: String = "changeme"

  /**
    * Is mutual SSL enabled.
    *
    */
  @BeanProperty
  @JsonProperty("mutual-enabled")
  var mutualEnabled: Boolean = false

  /**
    * Protocol to use for SSL encryption.
    *
    * For example: 'TLSv1.1', 'TLSv1.2'
    *
    */
  @BeanProperty
  @JsonProperty
  var protocol: String = "TLSv1.2"

  /**
    * (Docs taken from https://doc.akka.io/docs/akka/2.6/general/configuration-reference.html)
    * There are two options, and the default SecureRandom is recommended:
    * "" or "SecureRandom" => (default)
    * "SHA1PRNG" => Can be slow because of blocking issues on Linux
    *
    * "AES128CounterSecureRNG" and "AES256CounterSecureRNG" were deprecated in Akka
    * 2.5.16 and if these configuration values are used it will use the default
    * SecureRandom anyway and log a warning. The deprecated implementations are
    * not recommended, but they can be enabled by using configuration values
    * "DeprecatedAES128CounterSecureRNG" or "DeprecatedAES256CounterSecureRNG"
    *
    * Setting a value here may require you to supply the appropriate cipher
    * suite (see enabled-algorithms section above)
    *
    */
  @BeanProperty
  @JsonProperty("random-number-generator")
  var randomNumberGenerator: String = ""

  /**
    * Path to a trust store.
    *
    */
  @BeanProperty
  @JsonProperty("trust-store")
  var trustStore: String = "truststore"

  /**
    * Password for a key store.
    *
    */
  @BeanProperty
  @JsonProperty("trust-store-password")
  var trustStorePassword: String = "changeme"
}

class Events {

  /**
    * The time to live of events in the queue before it's going to be pruned.
    * As after that period of time the event can be outdated for the system and giving a not relevant information.
    *
    */
  @BeanProperty
  @JsonProperty("time-to-live")
  var timeToLive: String = "5 minutes"

  /**
    * The name of the queue where events related to CIs changes are sent.
    *
    */
  @BeanProperty
  @JsonProperty("cis-changed-queue-name")
  var cisChangedQueueName: String = "xld-cis-changed"

}

class Algorithms {
  /**
    * The algorithm for checksum computation, can be SHA-256 or SHA-1 (legacy)
    * SHA-256 is a default value starting from 9.5.0 release
    * SHA-256 should always be used for any new installation for security reasons
    * SHA-256 cannot be used for installations with existing file/folder/archive or any other artifacts with computed checksum from previous releases, use SHA-1 in this case
    *
    * The migration procedure will determine if SHA-256 can be used and will set SHA-1 otherwise
    *
    */
  @BeanProperty
  @JsonProperty
  var checksum: String = "SHA-256"
}

class PatchDictionaries {
  /**
    * The limitation for patch-dictionary file which can be uploaded (in Mb).
    *
    */
  @BeanProperty
  @JsonProperty("max-file-size-mb")
  var maxFileSize: Int = 5
}

class ExportCis {
  /**
   * The directory to use for the CLI `repository.exportCis()` function.
   *
   */
  @BeanProperty
  @JsonProperty("export-dir")
  @DocDefaultValue(value = "export", removeQuote = false)
  var exportDir: String = ServerSideProperties.DEFAULT_EXPORT_DIR

  /**
   * The directory to use for the CLI `repository.importCis()` function.
   *
   */
  @BeanProperty
  @JsonProperty("import-work-dir")
  @DocDefaultValue(value = "work", removeQuote = false)
  var importWorkDir: String = ServerSideProperties.DEFAULT_IMPORT_WORK_DIR
}

class Extensions {

  /**
    * ui extensions
    */
  @BeanProperty
  @JsonProperty("ui")
  var ui = new UiExtensionProperties

 /**
   * server extensions
   */
  @BeanProperty
  @JsonProperty("server")
  var server = new ServerExtensionProperties

}

class UiExtensionProperties {

  /**
    * The file where you specify the tabs (top menu items) you want to add in Deploy UI.
    * Each tab must be wired-up with some html file via the 'uri' property.
    */
  @BeanProperty
  @JsonProperty("file")
  @DocDefaultValue(value="xl-ui-plugin.xml")
  var file: String = ServerSideProperties.DEFAULT_DEPLOY_UI_EXTENSION_FILE

}

class ServerExtensionProperties {
  /**
    * File declares the endpoints that your extension adds to Deploy
    *
    */
  @BeanProperty
  @JsonProperty("file")
  var file: String = ServerSideProperties.DEFAULT_DEPLOY_SERVER_EXTENSION_FILE

  /**
    * Request timeout
    *
    */
  @BeanProperty
  @JsonProperty("timeout")
  var timeout: String = ServerSideProperties.DEFAULT_DEPLOY_SERVER_EXTENSION_TIMEOUT

  /**
    * Root path
    *
    */
  @BeanProperty
  @JsonProperty("rootPath")
  var rootPath: String = ServerSideProperties.DEFAULT_DEPLOY_SERVER_EXTENSION_ROOT_PATH

  /**
    * Script path prefix
    *
    */
  @BeanProperty
  @JsonProperty("scriptsPathPrefix")
  var scriptsPathPrefix: String = ServerSideProperties.DEFAULT_DEPLOY_SERVER_EXTENSION_SCRIPT_PATH_PREFIX
}

class Placeholders {
  /**
   * Should be set to true to enable placeholders archive.
   *
   */
  @BeanProperty
  @JsonProperty("archive")
  var archive: Boolean = false
}

class Analytics {
  /**
   * Specifies the capacity of the pendo queue
   *
   */
  @BeanProperty
  @JsonProperty("pendoQueueSize")
  var pendoQueueSize: Int = 1000

  /**
    * Specifies the cron trigger for pendo jobs
    * Each character corresponds to second (0-59), minute (0 - 59), hour (0 - 23), day of the month (1 - 31),
    * month (1 - 12) (or JAN-DEC), day of the week (0 - 7)(or MON-SUN)
    * "0 0 * * * *" = the top of every hour of every day.
    * "0 0 8-10 * * *" = 8, 9 and 10 o'clock of every day.
    * "0 0/30 8-10 * * *" = 8:00, 8:30, 9:00, 9:30, 10:00 and 10:30 every day.
    * "0 0 9-17 * * MON-FRI" = on the hour nine-to-five weekdays
    */
  @BeanProperty
  @JsonProperty("pendoMetricsCron")
  var pendoMetricsCron: String = "0 0 1 * * *"
}

@JsonInclude(Include.NON_NULL)
class SingletonScheduler {
  /**
    * Is Singleton Scheduler enabled or not.
    *
    */
  @BeanProperty
  @JsonProperty("enabled")
  var enabled: Boolean = true
}

class SupportPackageSettings {

  /**
    * Should be set to false to disable downloading application dependencies.
    *
    */
  @BeanProperty
  @JsonProperty("canDownloadAppDependencies")
  var canDownloadAppDependencies: Boolean = false

  /**
    * Should be set to false to disable downloading top deployed applications.
    *
    */
  @BeanProperty
  @JsonProperty("canDownloadTopDeployedApps")
  var canDownloadTopDeployedApps: Boolean = true

  /**
    * Should be set to false to disable downloading Environments and its members.
    *
    */
  @BeanProperty
  @JsonProperty("canDownloadEnvironments")
  var canDownloadEnvironments: Boolean = true

  /**
    * Should be set to false to disable downloading Descriptors.
    *
    */
  @BeanProperty
  @JsonProperty("canDownloadDescriptors")
  var canDownloadDescriptors: Boolean = true
}

object ServerSideProperties {

  var DEFAULT_WEB_CONTEXT_ROOT: String = "/"

  var DEFAULT_HTTP_PORT: Int = 4516

  var DEFAULT_HTTP_BIND_ADDRESS: String = "0.0.0.0"

  var DEFAULT_MAX_THREADS: Int = 150

  var DEFAULT_MIN_THREADS: Int = 30

  var KEY_HTTP_BIND_ADDRESS: String = "http.bind.address"

  var KEY_HTTP_PORT: String = "http.port"

  var KEY_HTTP_CONTEXT_ROOT: String = "http.context.root"

  var KEY_IMPORTABLE_PACKAGES_PATH: String = "importable.packages.path"

  var KEY_MINIMUM_THREADS: String = "threads.min"

  var KEY_MAXIMUM_THREADS: String = "threads.max"

  var KEY_ACTIVE_USER_SESSIONS: String = "active.user.sessions"

  var DEFAULT_EXPORT_DIR: String = "export"

  var DEFAULT_IMPORT_WORK_DIR: String = "work"

  var DEFAULT_MAX_PAGE_SIZE: Int = 1000

  var DEFAULT_DEPLOY_UI_EXTENSION_FILE = "xl-ui-plugin.xml"

  var DEFAULT_DEPLOY_SERVER_EXTENSION_FILE = "xl-rest-endpoints.xml"

  var DEFAULT_DEPLOY_SERVER_EXTENSION_ROOT_PATH = "/api"

  var DEFAULT_DEPLOY_SERVER_EXTENSION_SCRIPT_PATH_PREFIX = "/extension"

  var DEFAULT_DEPLOY_SERVER_EXTENSION_TIMEOUT = "60 seconds"

  var DEFAULT_SECURITY_AUTH_PROVIDER: String = "default"
}
