package ai.digital.configuration.central.deploy

import ai.digital.doc.properties.DocPropertiesFile
import com.fasterxml.jackson.annotation.JsonProperty
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

import java.beans.BeanProperty

/**
 * Enable and configure application level caching.
 * The caching is based on the standard JCache API(JSR 107).
 */

@Configuration
@ConfigurationProperties(prefix = "deploy.caches")
@DocPropertiesFile("deploy-caches.yaml")
class CachesProperties {

  /**
   * Toggle the overall caching at the application level
   */
  @BeanProperty
  var enabled: Boolean = false

  /**
   * Cache provider configuration
   */
  @BeanProperty
  @JsonProperty("provider-configuration")
  var providerConfiguration: ProviderConfiguration = new ProviderConfiguration

  /**
   * License cache configuration. Default ttl is 1440 minutes
   */
  @BeanProperty
  @JsonProperty("license-cache-configuration")
  var licenseCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 1440
  }

  /**
   * CI by pk cache configuration. Default ttl is 60 minutes
   */
  @BeanProperty
  @JsonProperty("ci-pk-cache-configuration")
  var ciPkCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 60
  }

  /**
   * CI by path cache configuration. Default ttl is 60 minutes
   */
  @BeanProperty
  @JsonProperty("ci-path-cache-configuration")
  var ciPathCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 60
  }

  /**
    * CI PROPERTIES cache configuration
    */
  @BeanProperty
  @JsonProperty("ci-properties-cache-configuration")
  var ciPropertiesCacheConfiguration: CacheConfiguration = new CacheConfiguration {
    ttlMinutes = 60
  }

  override def toString = s"CachesProperties(enabled=$enabled, providerConfiguration=$providerConfiguration, licenseCacheConfiguration=$licenseCacheConfiguration, ciPkCacheConfiguration=$ciPkCacheConfiguration, ciPathCacheConfiguration=$ciPathCacheConfiguration,  ciPropertiesCacheConfiguration=$ciPropertiesCacheConfiguration)"
}

class ProviderConfiguration {

  /**
   * Set the provider className to be passed to the Caching.getCachingProvider method.
   * An embedded Infinispan cache provider is used in single instance setup(single master with in-process worker) by default.
   * In all other setups a standalone cache server should be used and provider value should be configured.
   */
  @BeanProperty
  var provider: String = _

  /**
   * The provider specific configuration file name. This file should be on the classpath of all the
   * Deploy services(Master, Worker, Permission Service)
   */
  @BeanProperty
  @JsonProperty("provider-configuration-file")
  var providerConfigurationFile: String = _

  override def toString = s"ProviderConfiguration(provider=$provider, providerConfigurationFile=$providerConfigurationFile)"
}

class CacheConfiguration {
  /**
   * Toggle the individual cache
   */
  @BeanProperty
  var enabled: Boolean = true

  /**
   * Enable or disable management. If enabled the CacheMXBean for cache is registered
   * in the platform MBean server
   */
  @BeanProperty
  @JsonProperty("management-enabled")
  var managementEnabled: Boolean = true

  /**
   * Enable or disable statistics gathering for the cache
   */
  @BeanProperty
  @JsonProperty("statistics-enabled")
  var statisticsEnabled: Boolean = true

  /**
   * Time-to-live setting in minutes for entries in this cache
   */
  @BeanProperty
  @JsonProperty("ttl-minutes")
  var ttlMinutes: Int = 10

  override def toString = s"CacheConfiguration(enabled=$enabled, managementEnabled=$managementEnabled, statisticsEnabled=$statisticsEnabled, ttlMinutes=$ttlMinutes)"
}
