package ai.digital.configuration.central.deploy

import ai.digital.doc.properties.{DocDefaultValue, DocPropertiesFile}

import com.fasterxml.jackson.annotation.{JsonInclude, JsonProperty}
import com.fasterxml.jackson.annotation.JsonInclude.Include
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

import java.util.{Map => JMap}
import java.util.{List => JList}
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._

/**
  * Configurations related to artifacts managed by Deploy.
  */
@Configuration
@ConfigurationProperties(prefix = "deploy.artifact")
@JsonInclude(Include.NON_NULL)
@DocPropertiesFile("deploy-artifact-resolver.yaml")
class ArtifactResolverProperties {

  @BeanProperty
  @JsonProperty("resolver")
  var resolver = new Resolver

  @BeanProperty
  @JsonProperty("placeholders")
  var placeholders = new ArchiveExtensionsConfiguration

}

class Resolver {

  /**
    * HTTP Artifact resolver configuration
    */
  @BeanProperty
  @JsonProperty("http")
  var http = new Http

  /**
    * Maven Artifact resolver
    */
  @BeanProperty
  @JsonProperty("maven")
  var maven = new Maven
}

class Maven {

  /**
    * Switch to ignore missing artifacts - Deploy will serve and empty artifact if true is used.
    *
    */
  @BeanProperty
  @JsonProperty("ignoreMissingArtifact")
  var ignoreMissingArtifact: Boolean = false

  /**
    * Artifact resolver default working directory.
    *
    */
  @BeanProperty
  @JsonProperty("work-dir")
  var workDir: String = "work"

  /**
    * List of repositories, each repository contains the configuration related to it. This configuration includes:
    *
    * * Basic information: id and url
    * * authentication configuration with the same elements as servers have in settings.xml:
    * username, password, privateKey and passphrase.
    * * proxy configuration to use when connecting to this repository.
    * * Repository policies for releases and snapshots
    *
    */
  @BeanProperty
  @JsonProperty("repositories")
  @DocDefaultValue("none")
  var repositories: JList[Repositories] = JList.of()
}

@JsonInclude(Include.NON_NULL)
class Repositories {
  /**
    * Unique repository identifier.
    *
    * {{{for example: codehausSnapshots}}}
    *
    */
  @BeanProperty
  @JsonProperty("id")
  var id: String = _

  /**
    * Displayed repository name.
    *
    * {{{for example: Codehaus Snapshots}}}
    *
    */
  @BeanProperty
  @JsonProperty("name")
  var name: String = _

  /**
    * URL to the Maven repository
    *
    * {{{for example: http://snapshots.maven.codehaus.org/maven2}}}
    *
    */
  @BeanProperty
  @JsonProperty("url")
  var url: String = _

  /**
    * In case of private Maven repository, you have to specify credentials to be able to connect to it.
    *
    */
  @BeanProperty
  @JsonProperty("authentication")
  var authentication: Authentication = _

  /**
    * Proxy configuration if necessary.
    *
    */
  @BeanProperty
  @JsonProperty("proxy")
  var proxy: Proxy = _

  /**
    * Configuration to release repositories of Maven repository
    *
    */
  @BeanProperty
  @JsonProperty("releases")
  var releases = new Releases

  /**
    * Configuration to snapshot repositories of Maven repository
    *
    */
  @BeanProperty
  @JsonProperty("snapshots")
  var snapshots = new Snapshots

}

class Snapshots {
  /**
    * true or false for whether this repository is enabled for the respective type (releases or snapshots).
    *
    */
  @BeanProperty
  @JsonProperty("enabled")
  var enabled: Boolean = true

  /**
    * This element specifies how often updates should attempt to occur.
    * Maven will compare the local POM’s timestamp (stored in a repository’s maven-metadata file) to the remote.
    * The choices are: always, daily (default), interval:X (where X is an integer in minutes) or never.
    *
    */
  @BeanProperty
  @JsonProperty("updatePolicy")
  var updatePolicy: String = "daily"

  /**
    * When Maven deploys files to the repository, it also deploys corresponding checksum files.
    * Your options are to ignore, fail, or warn on missing or incorrect checksums.
    *
    */
  @BeanProperty
  @JsonProperty("checksumPolicy")
  var checksumPolicy: String = "warn"
}

class Releases {
  /**
    * true or false for whether this repository is enabled for the respective type (releases or snapshots).
    *
    */
  @BeanProperty
  @JsonProperty("enabled")
  var enabled: Boolean = true

  /**
    * This element specifies how often updates should attempt to occur.
    * Maven will compare the local POM’s timestamp (stored in a repository’s maven-metadata file) to the remote.
    * The choices are: always, daily (default), interval:X (where X is an integer in minutes) or never.
    *
    */
  @BeanProperty
  @JsonProperty("updatePolicy")
  var updatePolicy: String = "daily"

  /**
    * When Maven deploys files to the repository, it also deploys corresponding checksum files.
    * Your options are to ignore, fail, or warn on missing or incorrect checksums.
    *
    */
  @BeanProperty
  @JsonProperty("checksumPolicy")
  var checksumPolicy: String = "warn"

}

@JsonInclude(Include.NON_NULL)
class Proxy {

  /**
    * Proxy host. DNS or IP address.
    *
    */
  @BeanProperty
  @JsonProperty("host")
  var host: String = _

  /**
    * Proxy port
    *
    */
  @BeanProperty
  @JsonProperty("port")
  var port: Int = _

  /**
    * These elements appear as a pair denoting the login and password required to authenticate to this proxy server.
    *
    */
  @BeanProperty
  @JsonProperty("username")
  var username: String = _

  /**
    * These elements appear as a pair denoting the login and password required to authenticate to this proxy server.
    *
    */
  @BeanProperty
  @JsonProperty("password")
  var password: String = _

}

@JsonInclude(Include.NON_NULL)
class Authentication {
  /**
    * These elements appear as a pair denoting the login and password required to authenticate to this server.
    *
    */
  @BeanProperty
  @JsonProperty("username")
  var username: String = _

  /**
    * These elements appear as a pair denoting the login and password required to authenticate to this server.
    *
    */
  @BeanProperty
  @JsonProperty("password")
  var password: String = _

  /**
    * It's possible to connect via private ssh key.
    *
    * {{{for example: (/Users/mavenUser/.ssh/id_dsa)}}}
    *
    */
  @BeanProperty
  @JsonProperty("privateKey")
  var privateKey: String = _

  /**
    * The pass phrase for the private key
    *
    * {{{for example: some_passphrase}}}
    */
  @BeanProperty
  @JsonProperty("passphrase")
  var passphrase: String = _
}


class Http {
  /**
    * Switch to ignore missing artifacts - Deploy will serve and empty artifact if true is used
    *
    */
  @BeanProperty
  @JsonProperty("ignoreMissingArtifact")
  var ignoreMissingArtifact: Boolean = false
}

class ArchiveExtensionsConfiguration {
  /**
    * Configuration which stream to use for which extension.
    *
    * Functionality is built on top of "Apache Commons Compress" library.
    *
    * Zip -> ZipFileArchiveStreamer
    *
    * Jar -> JarFileArchiveStreamer
    *
    * Tar, tar.gz, tar.bz2 -> ArchiveStreamer
    *
    */
  @BeanProperty
  @JsonProperty("archive-extensions")
  @DocDefaultValue(value =
    "zip -> 'zip',\njar-> 'jar',\ntar-> 'tar',\n'tar.gz' -> 'tar.gz',\n'tar.bz2'-> 'tar.bz2',\near -> 'jar',\nwar -> 'jar',\nsar -> 'jar',\nrar -> 'jar',\naop -> 'jar',\nhar -> 'jar',\naar -> 'zip'",
    removeQuote = true)
  var archiveExtensions: JMap[String, String] = Map(
    "zip" -> "zip",
    "tar.gz" -> "tar.gz",
    "rar" -> "jar",
    "aop" -> "jar",
    "tar" -> "tar",
    "tar.bz2" -> "tar.bz2",
    "sar" -> "jar",
    "ear" -> "jar",
    "har" -> "jar",
    "war" -> "jar",
    "jar" -> "jar"
  ).asJava

  /**
    * Encoding used for decoding filenames in the ZIP files. Possible values are UTF8 and CP347. When CP347 is used decoding of the
    * filenames will work according to the ZIP specification (obeying EFS bit).
    *
    * Default value is: UTF8
    */
  @BeanProperty
  @JsonProperty("zip-encoding")
  var zipEncoding: String = "UTF8"

  /**
    * Encoding used for decoding filenames in the JAR files. Possible values are UTF8 and CP347. When CP347 is used decoding of the
    * filenames will work according to the ZIP specification (obeying EFS bit).
    *
    * Default value is: UTF8
    */
  @BeanProperty
  @JsonProperty("jar-encoding")
  var jarEncoding: String = "UTF8"
}
