package ai.digital.configuration.central.deploy

import ai.digital.doc.properties.{DocDefaultValue, DocPropertiesFile}
import com.fasterxml.jackson.annotation.JsonInclude.Include
import com.fasterxml.jackson.annotation.{JsonInclude, JsonProperty}
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

import java.util.{List => JList}
import scala.beans.BeanProperty

/**
  * This Configuration is used to enable identity providers that support the OIDC authentication protocol,
  * such as OKTA, Keycloak, and Azure Active Directory (Office 365).
  * In the above identities, any one single authentication option will be supported.
  */
@Configuration
@ConfigurationProperties(prefix = "deploy.security")
@JsonInclude(Include.NON_NULL)
@DocPropertiesFile("deploy-oidc.yaml")
class OidcProperties {

  /**
    * Setting for Authentication.
    */
  @BeanProperty
  @JsonProperty("auth")
  var auth = new Auth

}

class Auth {
  /**
    * Setting for Authentication Provider.
    */
  @BeanProperty
  @JsonProperty("providers")
  var providers = new Providers
}

class Providers {
  /**
    * Setting for oidc Configuration.
    */
  @BeanProperty
  @JsonProperty("oidc")
  var oidc = new Oidc
}

@JsonInclude(Include.NON_NULL)
class Oidc {
  /**
    * Login Method Description. Default to Login Method.
    * {{{
    * for example - "Login using Keycloak"
    * }}}
    *
    */
  @BeanProperty
  @JsonProperty("loginMethodDescription")
  var loginMethodDescription: String = "Login Method"

  /**
    * An identifier (ID) of the service associated with the client in Hub.
    *
    * {{{
    *   for example - "deploy"
    * }}}
    *
    */
  @BeanProperty
  @JsonProperty("clientId")
  var clientId: String = _

  /**
    * A secret for the service associated with the client in Hub.
    *
    * {{{
    *   for example - "ff8a8547-dbe3-4267-ae55-9822d6f02499"
    * }}}
    *
    */
  @BeanProperty
  @JsonProperty("clientSecret")
  var clientSecret: String = _

  /**
    * Optional.
    *
    * JSON array containing a list of Client Authentication methods supported by this Token Endpoint.
    * The options are client_secret_post, client_secret_basic, client_secret_jwt, and private_key_jwt,
    * as described in Section 9 of OpenID Connect Core 1.0 [OpenID.Core].
    * Other authentication methods MAY be defined by extensions.
    * If omitted, the default is client_secret_basic -- the HTTP
    * Basic Authentication Scheme specified in Section 2.3.1 of OAuth 2.0 [RFC6749].
    *
    * {{{
    *   for example - "client_secret_post"/"client_secret_basic"
    * }}}
    *
    */
  @BeanProperty
  @JsonProperty("clientAuthMethod")
  var clientAuthMethod: String = _

  /**
    * URL using the https scheme with no query or fragment component that the OP asserts as its Issuer Identifier.
    * If Issuer discovery is supported (see Section 2), this value MUST be identical to the issuer value returned by WebFinger.
    * This also MUST be identical to the iss Claim value in ID Tokens issued from this Issuer.
    *
    * {{{
    *   for example - "http://localhost/auth/realms/digitalai-platform"
    * }}}
    *
    */
  @BeanProperty
  @JsonProperty("issuer")
  var issuer: String = _

  /**
    * The key retrieval uri.
    * {{{for example - "https://oidc.example.com/endpoint/keys"}}}
    *
    */
  @BeanProperty
  @JsonProperty("keyRetrievalUri")
  var keyRetrievalUri: String = _

  /**
    * The redirect URI to use for returning the access token.
    *
    * {{{for example - https://example.oktapreview.com/oauth2/sometoken/v1/token}}}
    *
    */
  @BeanProperty
  @JsonProperty("accessTokenUri")
  var accessTokenUri: String = _

  /**
    * The authorize endpoint to request tokens or authorization codes via the browser.
    *
    * {{{for example - https://example.oktapreview.com/oauth2/sometoken/v1/authorize}}}
    *
    */
  @BeanProperty
  @JsonProperty("userAuthorizationUri")
  var userAuthorizationUri: String = _

  /**
    * The logout endpoint to revoke token via the browser.
    *
    * {{{for example - https://example.oktapreview.com/oauth2/sometoken/v1/logout}}}
    *
    */
  @BeanProperty
  @JsonProperty("logoutUri")
  var logoutUri: String = _

  /**
    * The uri for the redirection endpoint.
    *
    * {{{for example - "https://xl-deploy.example.com/login/external-login"}}}
    *
    */
  @BeanProperty
  @JsonProperty("redirectUri")
  var redirectUri: String = _

  /**
    * The redirect to the login page after logout.
    *
    * {{{
    *   for example - "https://xl-deploy.example.com/login/external-login".
    * }}}
    */
  @BeanProperty
  @JsonProperty("postLogoutRedirectUri")
  var postLogoutRedirectUri: String = _

  /**
    * In Deploy, the OIDC roles become principals that you can assign to roles inside Deploy.
    *
    */

  @BeanProperty
  @JsonProperty("rolesClaimName")
  var rolesClaimName: String = _

  /**
    * Unique username for both internal and external users.
    * You cannot sign in with a user if a local account with the same username exists.
    *
    */

  @BeanProperty
  @JsonProperty("userNameClaimName")
  var userNameClaimName: String = _

  /**
    * Digital Signature or MAC Algorithm
    *
    * ------------------------------------------
    * |Value|Digital Signature or MAC Algorithm|
    * |RS256 |	RSASSA-PKCS1-v1_5 using SHA-256|
    * |RS384 | 	RSASSA-PKCS1-v1_5 using SHA-384|
    * |RS512 | RSASSA-PKCS1-v1_5 using SHA-512|
    * |ES256 | ECDSA using P-256 and SHA-256|
    * |ES384	| ECDSA using P-384 and SHA-384|
    * |ES512	| ECDSA using P-521 and SHA-512|
    * |PS256	| RSASSA-PSS using SHA-256 and MGF1 with SHA-256|
    * |PS384	| RSASSA-PSS using SHA-384 and MGF1 with SHA-384|
    * |PS512	| RSASSA-PSS using SHA-512 and MGF1 with SHA-512|
    * |HS256	| HMAC using SHA-256|
    * |HS384	| HMAC using SHA-384|
    * |HS512	| HMAC using SHA-512|
    *
    */
  @BeanProperty
  @JsonProperty("idTokenJWSAlg")
  var idTokenJWSAlg: String = _

  /**
    * Settings for AccessToken.
    *
    */
  @BeanProperty
  @JsonProperty("access-token")
  var accessToken: AccessToken = _

  /**
    * The host name of the proxy server.
    *
    */
  @BeanProperty
  @JsonProperty("proxyHost")
  var proxyHost: String = _

  /**
    * The port number of the proxy server.
    *
    */
  @BeanProperty
  @JsonProperty("proxyPort")
  var proxyPort: Integer = _

  /**
    * oidc scopes.
    * {{{for example - [openid]}}}
    *
    */
  @BeanProperty
  @JsonProperty("scopes")
  @DocDefaultValue(value="[]", removeQuote = true)
  var scopes: JList[String] = JList.of()
}

@JsonInclude(Include.NON_NULL)
class AccessToken {
  /**
    * The Access Token issuer.
    * Expected issuer 'iss' claim value.
    *
    */
  @BeanProperty
  @JsonProperty("issuer")
  var issuer: String = _

  /**
    * Expected audience 'aud' claim value.
    *
    */
  @BeanProperty
  @JsonProperty("audience")
  var audience: String = _

  /**
    * The jwks_uri to retrieve keys for the token.
    *
    */
  @BeanProperty
  @JsonProperty("keyRetrievalUri")
  var keyRetrievalUri: String = _

  /**
    * The token signature verification algorithm.
    *
    */
  @BeanProperty
  @JsonProperty("jwsAlg")
  var jwsAlg: String = _

  /**
    * The Secret.
    * The secret key if MAC based algorithms is used for the token.
    *
    */
  @BeanProperty
  @JsonProperty("secretKey")
  var secretKey: String = _
}
