package ai.digital.configuration.central.deploy

import ai.digital.doc.properties.{DocDefaultValue, DocPropertiesFile}
import com.fasterxml.jackson.annotation.JsonProperty
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.stereotype.Component

import java.beans.BeanProperty
import java.util.{List => JList}

/**
  * This configuration is used during deployments to prevent users to perform executing some commands
  * Or allowing to perform certain commands for security reasons.
  * {{{
  * for example :
  *  a) We have two user, user1 and user2 and user1 has the developer role.
  *  b) Create the Command Plugin application(CommandApp01) with following command CI type.
  *     - "ls"
  *     - "date"
  *
  *  1) When user1 try to do deployment CommandApp01.
  *      Deployment will be success, because it as the allowed set of commands from both all-users and roles.
  *  2) When user2 try to do deployment of CommandApp01.
  *     Deployment fails because "date" is not in allowed command for all users.
  *
  *   deploy.command-whitelist:
  *   enabled: true
  *   all-users:
  *     allowed-commands: ["^.*ls.*$"]
  *     restricted-commands: []
  *   roles:
  *     - role-name: "developer"
  *       allowed-commands: ["^.*date.*$"]
  *       restricted-commands: []
  *
  * We need to configure either allowed command or restricted commands, only one needs to be set at any time.
  * if both are configured deployment will fail with following statement
  *  "Both allowed commands and restricted commands specified for all users. Please set only one of those properties."
  * }}}
  *
  */
@Component("commandWhitelistProperties")
@ConfigurationProperties(prefix = "deploy.command-whitelist")
@DocPropertiesFile("deploy-command-whitelist.yaml")
class CommandWhitelistProperties {

  /**
    * Property to enable the whitelist commands, which can be used during deployment. Default to false
    */
  @BeanProperty
  @DocDefaultValue(value="false", removeQuote = true)
  var enabled: Boolean = _

  /**
    * Property that Specifies list of command which we can be allowed or restricted for all the users.
    * {{{
    * for example:
    *   1) Any user, can do the deployment which has the "date" command, for other commands deployment fails.
    *    all-users:
    *         allowed-commands: ["^.*date.*$"]
    *         restricted-commands: [ ]
    *   2) Any user, if they try to do deployment which has the "rm" command,
    *      deployment fails with statement "Command rm is restricted by rule ^.*rm.*$" for all users.
    *       all-users:
    *          allowed-commands: []
    *          restricted-commands: ["^.*rm.*$" ]
    * }}}
    */
  @BeanProperty
  @JsonProperty("all-users")
  var allUsers: AllUsers = _

  /**
    * Property that Specifies list of command which we can be allowed or restricted based on roles specified.
    * {{{
    * for example:
    *   1) user who has the developer role, can do the deployment which has the "date" command, for other commands deployment fails.
    *       roles:
    *        - role-name: "developer"
    *          allowed-commands: ["^.*date.*$"]
    *          restricted-commands: []
    *   2) user who has the developer role, if they try to deployment which has the "rm" command,
    *   deployment fails with statement "Command rm is restricted for role developer by rule ^.*rm.*$".
    *       roles:
    *        - role-name: "developer"
    *          allowed-commands: []
    *          restricted-commands: ["^.*rm.*$" ]
    * }}}
    */
  @BeanProperty
  @DocDefaultValue(value="[]", removeQuote = true)
  var roles: JList[Roles] = JList.of()
}

class AllUsers {

  /**
    * The Allowed Commands.
    * List of commands that can be allowed in command plugin.
    * {{{
    *   for example :
    *     - ls
    *     - ^.*ls.*$ [using regex]
    * }}}
    */
  @BeanProperty
  @JsonProperty("allowed-commands")
  @DocDefaultValue(value="[]", removeQuote = true)
  var allowedCommands: JList[String] = JList.of()

  /**
    * The Restricted Commands.
    * List of commands that can be restricted in command plugin.
    * {{{
    *   for example :
    *     - ls
    *     - ^.*ls.*$ [using regex]
    * }}}
    */
  @BeanProperty
  @JsonProperty("restricted-commands")
  @DocDefaultValue(value="[]", removeQuote = true)
  var restrictedCommands: JList[String] = JList.of()

}

class Roles {

  /**
    * The role name
    * {{{for example - Developer}}}
    */
  @BeanProperty
  var roleName: String = _

  /**
    * The Allowed Commands.
    * List of commands that can be allowed in command plugin.
    * {{{
    *   for example :
    *     - ls
    *     - ^.*ls.*$ [using regex]
    * }}}
    */
  @BeanProperty
  @JsonProperty("allowed-commands")
  @DocDefaultValue(value="[]", removeQuote = true)
  var allowedCommands: JList[String] = JList.of()

  /**
    * The Restricted Commands.
    * List of commands that can be restricted in command plugin.
    * {{{
    *   for example :
    *     - ls
    *     - ^.*ls.*$ [using regex]
    * }}}
    */
  @BeanProperty
  @JsonProperty("restricted-commands")
  @DocDefaultValue(value="[]", removeQuote = true)
  var restrictedCommands: JList[String] = JList.of()

}
