package ai.digital.configuration.central.deploy

import ai.digital.configuration.central.deploy.converter.HoconDurationConverter
import ai.digital.doc.properties.{DocDefaultValue, DocPropertiesFile}

import java.util.{List => JList}
import com.fasterxml.jackson.annotation.JsonInclude.Include
import com.fasterxml.jackson.annotation.{JsonInclude, JsonProperty}
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.{Bean, Configuration}

import java.util.concurrent.TimeUnit
import scala.beans.BeanProperty

/**
 * Configurations related to the task engine.
 * It is responsible for running deployments.
 */
@Configuration
@ConfigurationProperties(prefix = "deploy.task")
@JsonInclude(Include.NON_NULL)
@DocPropertiesFile("deploy-task.yaml")
class TaskerSystemProperties {

  /**
   * Should the system use in process worker or connect to external workers.
   * In case of true, the system will be running (deployment, control) tasks on a master node.
   *
   */
  @BeanProperty
  @JsonProperty("in-process-worker")
  var inProcessWorker: Boolean = true

  /**
   * The maximum active tasks which system will keep for the execution.
   * In case of exceeding this limit, next task will be immediately queued.
   *
   */
  @BeanProperty
  @JsonProperty("max-active-tasks")
  var maxActiveTasks: Int = 100

  @BeanProperty
  @JsonProperty("artifact-copy-strategy")
  var artifactCopyStrategy = new ArtifactCopyStrategy

  /**
   * The task recovery directory location.
   * This path can be relative (to Deploy home installation) and also absolute.
   *
   */
  @BeanProperty
  @JsonProperty("recovery-dir")
  var recoveryDir: String = "work"

  /**
   * Timeout to gracefully shut down the system.
   *
   * If after this period of time the system will be exited forcefully.
   */
  @BeanProperty
  @JsonProperty("shutdown-timeout")
  var shutdownTimeout: String = "10 minutes"

  /**
    * The queue configuration of the type system.
    */
  @BeanProperty
  @JsonProperty("events")
  var events: TaskEvents = new TaskEvents

  /**
   * The queue configuration of the type system.
   */
  @BeanProperty
  @JsonProperty("queue")
  var queue: Queue = new Queue

  @BeanProperty
  @JsonProperty("step")
  var step = new Step

  @BeanProperty
  @JsonProperty("logger")
  var logger = new Logger

  @BeanProperty
  @JsonProperty("system")
  var taskSystem = new TaskSystem

  @BeanProperty
  @JsonProperty("dispatchers")
  var dispatchers = new Dispatchers

  @BeanProperty
  @JsonProperty("scheduler")
  var scheduler: SchedulerConfiguration = new SchedulerConfiguration

  @BeanProperty
  @JsonProperty("planner")
  var planner: PlannerConfiguration = new PlannerConfiguration

  /**
    * For testing purpose and future use only. Enables saving of the task path statuses to the database.
    */
  @BeanProperty
  @JsonProperty("resilient")
  var resilient: Boolean = false

  @Bean
  def taskQueueBrokerConfig: TaskQueueBrokerConfig = {
    if (inProcessWorker)
      InProcessConfig(queue.inProcess.maxDiskUsage, queue.inProcess.shutdownTimeout)
    else
      ExternalConfig(
        queue.external.jmsDriverClassname,
        queue.external.jmsUrl,
        queue.external.jmsUsername,
        queue.external.jmsPassword)
  }
}

class TaskEvents {

  @BeanProperty
  @JsonProperty("task-path-status")
  var taskPathStatus: TaskPathStatus = new TaskPathStatus
}

class TaskPathStatus {

  @BeanProperty
  @JsonProperty("queue")
  var queue: EventsQueue = new EventsQueue
}

class EventsQueue {

  /**
    * The time to live of events in the queue before it's going to be pruned.
    * As after that period of time the event can be outdated for the system and giving a not relevant information.
    */
  @BeanProperty
  @JsonProperty("time-to-live")
  var timeToLive: String = "5 minutes"

  /**
    * The name of the queue where are posted events for the task path status changes.
    */
  @BeanProperty
  @JsonProperty("name")
  var name: String = "xld-task-path-status"

  /**
    * Before sending messages to the queue, we collect them in an internal map.
    *
    * The number of events defined by this property are composed to 1 message.
    *
    * This message is going to be sent based on 2 parameters:
    * - event-flush-period reached the period
    * - number of messages (event-flush-threshold) exceeded defined threshold.
    *
    * The receiving part operates also with the batch of events and insert it to a database.
    *
    * This is done for the sake of performance.
    */
  @BeanProperty
  @JsonProperty("event-flush-period")
  var eventFlushPeriod = "1 second"

  /**
    * Before sending messages to the queue, we collect them in an internal map.
    *
    * The number of events defined by this property are composed to 1 message.
    * This message is going to be sent based on 2 parameters:
    * - event-flush-period reached the period
    * - number of messages (event-flush-threshold) exceeded defined threshold.
    *
    * The receiving part operates also with the batch of events and insert it to a database.
    *
    * This is done for the sake of performance.
    */
  @BeanProperty
  @JsonProperty("event-flush-threshold")
  var eventFlushThreshold = 1000

  def getEventFlushPeriodInMillis: Long = HoconDurationConverter.convert(eventFlushPeriod, TimeUnit.MILLISECONDS)

}

class Dispatchers {
  @BeanProperty
  @JsonProperty("state-management-dispatcher")
  var stateManagementDispatcher = new StateManagementDispatcher

  @BeanProperty
  @JsonProperty("step-dispatcher")
  var stepDispatcher = new StepDispatcher

  @BeanProperty
  @JsonProperty("state-event-listener-dispatcher")
  var stateEventListenerDispatcher = new StateEventListenerDispatcher

  @BeanProperty
  @JsonProperty("archive-dispatcher")
  var archiveDispatcher = new ArchiveDispatcher
}

@JsonInclude(Include.NON_EMPTY)
class ArchiveDispatcher {
  /**
   * An Akka MessageDispatcher is what makes Akka Actors “tick”, it is the engine of the machine so to speak.
   * All MessageDispatcher implementations are also an Executor, which means that they can be used to execute arbitrary code,
   * for instance CompletableFutures.
   *
   * There are 2 different types of message dispatchers:
   *
   * 1. Dispatcher
   *   This is an event-based dispatcher that binds a set of Actors to a thread pool. The default dispatcher is used if no other is specified.
   * 2. PinnedDispatcher
   *   This dispatcher dedicates a unique thread for each actor using it;
   *   i.e. each actor will have its own thread pool with only one thread in the pool.
   *
   */
  @BeanProperty
  @JsonProperty("type")
  var dispatcherType: String = "Dispatcher"

  /**
   * What kind of ExecutionService to use.
   *
   */
  @BeanProperty
  @JsonProperty("executor")
  var executor: String = "fork-join-executor"

  /**
   * Throughput defines the maximum number of messages to be
   * processed per actor before the thread jumps to the next actor.
   * Set to 1 for as fair as possible.
   *
   */
  @BeanProperty
  @JsonProperty("throughput")
  var throughput: Int = 100

  @BeanProperty
  @JsonProperty("fork-join-executor")
  var forkjoinExecutor = new {
    @BeanProperty
    @JsonProperty("parallelism-min")
    var parallelismMin: Int = 2

    /**
     * Parallelism (threads) ... ceil(available processors * factor).
     *
     */
    @BeanProperty
    @JsonProperty("parallelism-factor")
    var parallelismFactor: Double = 2.0

    /**
     * Maximum number of threads to cap factor-based parallelism number to.
     *
     */
    @BeanProperty
    @JsonProperty("parallelism-max")
    var parallelismMax: Int = 16
  }
}

@JsonInclude(Include.NON_EMPTY)
class StateEventListenerDispatcher {
  /**
   * An Akka MessageDispatcher is what makes Akka Actors “tick”, it is the engine of the machine so to speak.
   * All MessageDispatcher implementations are also an Executor, which means that they can be used to execute arbitrary code,
   * for instance CompletableFutures.
   *
   * There are 2 different types of message dispatchers:
   *
   * 1. Dispatcher
   *   This is an event-based dispatcher that binds a set of Actors to a thread pool. The default dispatcher is used if no other is specified.
   * 2. PinnedDispatcher
   *   This dispatcher dedicates a unique thread for each actor using it;
   *   i.e. each actor will have its own thread pool with only one thread in the pool.
   *
   */
  @BeanProperty
  @JsonProperty("type")
  var dispatcherType: String = "Dispatcher"

  /**
   * What kind of ExecutionService to use.
   *
   */
  @BeanProperty
  @JsonProperty("executor")
  var executor: String = "fork-join-executor"

  /**
   * Throughput defines the maximum number of messages to be
   * processed per actor before the thread jumps to the next actor.
   * Set to 1 for as fair as possible.
   *
   */
  @BeanProperty
  @JsonProperty("throughput")
  var throughput: Int = 100

  @BeanProperty
  @JsonProperty("fork-join-executor")
  var forkjoinExecutor = new {
    /**
     * Minimum number of threads to cap factor-based parallelism number to.
     *
     */
    @BeanProperty
    @JsonProperty("parallelism-min")
    var parallelismMin: Int = 2

    /**
     * Parallelism (threads) ... ceil(available processors * factor).
     *
     */
    @BeanProperty
    @JsonProperty("parallelism-factor")
    var parallelismFactor: Double = 2.0

    /**
     * Maximum number of threads to cap factor-based parallelism number to.
     *
     */
    @BeanProperty
    @JsonProperty("parallelism-max")
    var parallelismMax: Int = 16
  }
}

@JsonInclude(Include.NON_EMPTY)
class StepDispatcher {
  /**
   * An Akka MessageDispatcher is what makes Akka Actors “tick”, it is the engine of the machine so to speak.
   * All MessageDispatcher implementations are also an Executor, which means that they can be used to execute arbitrary code,
   * for instance CompletableFutures.
   *
   * There are 2 different types of message dispatchers:
   *
   * 1. Dispatcher
   *   This is an event-based dispatcher that binds a set of Actors to a thread pool. The default dispatcher is used if no other is specified.
   * 2. PinnedDispatcher
   *   This dispatcher dedicates a unique thread for each actor using it;
   *   i.e. each actor will have its own thread pool with only one thread in the pool.
   *
   */
  @BeanProperty
  @JsonProperty("type")
  var dispatcherType: String = "Dispatcher"

  /**
   * What kind of ExecutionService to use.
   *
   */
  @BeanProperty
  @JsonProperty("executor")
  var executor: String = "thread-pool-executor"

  /**
   * Throughput defines the maximum number of messages to be
   * processed per actor before the thread jumps to the next actor.
   * Set to 1 for as fair as possible.
   *
   */
  @BeanProperty
  @JsonProperty("throughput")
  var throughput: Int = 100

  @BeanProperty
  @JsonProperty("thread-pool-executor")
  var threadPoolExecutor = new {

    /**
     * Minimum number of threads to cap factor-based core number to.
     *
     */
    @BeanProperty
    @JsonProperty("core-pool-size-min")
    var corePoolSizeMin: Int = 32

    /**
     * No of core threads ... ceil(available processors * factor).
     *
     */
    @BeanProperty
    @JsonProperty("core-pool-size-factor")
    var corePoolSizeFactor: Double = 3.0

    /**
     * Maximum number of threads to cap factor-based number to.
     *
     */
    @BeanProperty
    @JsonProperty("core-pool-size-max")
    var corePoolSizeMax: Int = 32

    /**
     * Specifies the bounded capacity of the task queue (< 1 == unbounded)
     *
     * Keep the queue unbound to avoid that we have to deal with rejections.
     *
     */
    @BeanProperty
    @JsonProperty("task-queue-size")
    var taskQueueSize: Int = -1
  }
}

@JsonInclude(Include.NON_EMPTY)
class StateManagementDispatcher {
  /**
   * An Akka MessageDispatcher is what makes Akka Actors “tick”, it is the engine of the machine so to speak.
   * All MessageDispatcher implementations are also an Executor, which means that they can be used to execute arbitrary code,
   * for instance CompletableFutures.
   *
   * There are 2 different types of message dispatchers:
   *
   * 1. Dispatcher
   *   This is an event-based dispatcher that binds a set of Actors to a thread pool. The default dispatcher is used if no other is specified.
   * 2. PinnedDispatcher
   *   This dispatcher dedicates a unique thread for each actor using it;
   *   i.e. each actor will have its own thread pool with only one thread in the pool.
   *
   */
  @BeanProperty
  @JsonProperty("type")
  var dispatcherType: String = "Dispatcher"

  /**
   * What kind of ExecutionService to use.
   *
   */
  @BeanProperty
  @JsonProperty("executor")
  var executor: String = "fork-join-executor"

  /**
   * Throughput defines the maximum number of messages to be
   * processed per actor before the thread jumps to the next actor.
   * Set to 1 for as fair as possible.
   *
   */
  @BeanProperty
  @JsonProperty("throughput")
  var throughput: Int = 100

  @BeanProperty
  @JsonProperty("fork-join-executor")
  var forkjoinExecutor = new {
    /**
     * Minimum number of threads to cap factor-based parallelism number to.
     *
     */
    @BeanProperty
    @JsonProperty("parallelism-min")
    var parallelismMin: Int = 8

    /**
     * Parallelism (threads) ... ceil(available processors * factor).
     *
     */
    @BeanProperty
    @JsonProperty("parallelism-factor")
    var parallelismFactor: Double = 2.0

    /**
     * Maximum number of threads to cap factor-based parallelism number to.
     *
     */
    @BeanProperty
    @JsonProperty("parallelism-max")
    var parallelismMax: Int = 32
  }

}

class TaskSystem {
  @BeanProperty
  @JsonProperty("akka")
  var akka = new SystemAkka
}

@JsonInclude(Include.NON_EMPTY)
class SystemAkka {
  /**
   * Loggers to register at boot time.
   *
   */
  @BeanProperty
  @JsonProperty("loggers")
  @DocDefaultValue(value="[\"akka.event.slf4j.Slf4jLogger\"]", removeQuote = true)
  var loggers: JList[String] = JList.of("akka.event.slf4j.Slf4jLogger")

  /**
   * Options: OFF, ERROR, WARNING, INFO, DEBUG.
   *
   */
  @BeanProperty
  @JsonProperty("loglevel")
  var loglevel: String = "INFO"

  /**
   * To be able to load extensions from your Akka configuration you must add FQCNs of implementations of either
   * ExtensionId or ExtensionIdProvider in the akka.extensions section of the config you provide to your ActorSystem.
   */
  @BeanProperty
  @JsonProperty("extensions")
  @DocDefaultValue(value="[]", removeQuote = true)
  var extensions: JList[String] = JList.of()

  /**
   * Don't use jvm-exit-on-fatal-error=off. It's safer to shutdown the JVM in case of a fatal error, such as OutOfMemoryError.
   * Related config properties: [akka.jvm-exit-on-fatal-error = off].
   * Corresponding default values: [akka.jvm-exit-on-fatal-error = on].
   * You may disable this check by adding [jvm-exit-on-fatal-error] to configuration string list akka.diagnostics.checker.disabled-checks.
   *
   */
  @BeanProperty
  @JsonProperty("jvm-exit-on-fatal-error")
  var jvmExitOnFatalError: String = "on"

  @BeanProperty
  @JsonProperty("actor")
  var actor = new SystemAkkaActor
}

class SystemAkkaActor {
  /**
   * Akka actor debug configurations.
   */
  @BeanProperty
  @JsonProperty("debug")
  var debug = new SystemAkkaActorDebug
}

class SystemAkkaActorDebug {
  /**
   * Will log all messages sent to an actor if that actors receive method is a LoggingReceive.
   *
   */
  @BeanProperty
  @JsonProperty("receive")
  var receive: String = "on"

  /**
   * Enables DEBUG logging of all AutoReceiveMessages (Kill, PoisonPill etc.)
   *
   */
  @BeanProperty
  @JsonProperty("autoreceive")
  var autoReceive: String = "on"

  /**
   * Enables DEBUG logging of actor lifecycle changes.
   *
   */
  @BeanProperty
  @JsonProperty("lifecycle")
  var lifecycle: String = "on"
}

class ArtifactCopyStrategy {

  /**
   * If true, copy strategy will auto-detect the strategy by extension.
   * Default strategy for artifact archives is to copy files one by one, what might affect the performance.
   * Even if by default this property as false, it is still possible to override the copy
   * strategy on a container level.
   *
   */
  @BeanProperty
  @JsonProperty("autodetect")
  var autodetect: Boolean = false
}

class Logger {

  /**
   * Configuration to enable Elastic4s.
   *
   * Elastic4s is a concise, idiomatic, reactive, type safe Scala client for Elasticsearch
   */
  @BeanProperty
  @JsonProperty("elastic")
  var elastic = new {

    /**
     * URI to Elasticsearch server.
     */
    @BeanProperty
    @JsonProperty("uri")
    var uri: String = _

    /**
     * Name of the index created in ElasticSearch.
     *
     */
    @BeanProperty
    @JsonProperty("index")
    var index: String = "xld-log"
  }
}

class Step {
  /**
   * How many threads to keep for step execution.
   *
   */
  @BeanProperty
  @JsonProperty("execution-threads")
  var executionThreads: Int = 32

  /**
   * Delay between each retry.
   *
   */
  @BeanProperty
  @JsonProperty("retry-delay")
  var retryDelay: String = "5 seconds"

  /**
   * The maximum allowed step run.
   * It is helpful to detect hanging steps.
   *
   */
  @BeanProperty
  @JsonProperty("run-timeout")
  var runTimeout: String = "6 hours"

  /**
   * In case of enabled retry policy, how many times to attempt the retry.
   *
   */
  @BeanProperty
  @JsonProperty("max-retry-number")
  var maxRetryNumber: Int = 4320

  @BeanProperty
  @JsonProperty("on-copy-artifact")
  var onCopyArtifact = new OnCopyArtifact
}

class OnCopyArtifact {
  /**
   * This option enables the retry in case of failure of artifact copy step.
   *
   * It can be helpful in situations with a flaky network, to prevent manual intervention to restart the task.
   *
   */
  @BeanProperty
  @JsonProperty("enable-retry")
  var enableRetry: Boolean = true
}

class InProcess {
  /**
   * This configuration is specific to an internal JMS queue.
   *
   * The max percentage of data used use from disks.
   * The System will block when the disk usage exceeds this value.
   *
   */
  @BeanProperty
  @JsonProperty("maxDiskUsage")
  var maxDiskUsage: Int = 100

  /**
   * This configuration is specific to an internal JMS queue.
   *
   * Timeout (in ms) on waiting for clients to disconnect before server shutdown.
   *
   */
  @BeanProperty
  @JsonProperty("shutdownTimeout")
  var shutdownTimeout: Long = 600000
}

@JsonInclude(Include.NON_EMPTY)
class Queue {
  /**
   * The name of the queue where all created tasks IDs are sent with the consequent processing by task engine.
   *
   */
  @BeanProperty
  @JsonProperty("name")
  var name = "xld-tasks-queue"

  /**
   * The name of the queue where all requests published to archive the tasks.
   *
   */
  @BeanProperty
  @JsonProperty("archive-queue-name")
  var archiveQueueName = "xld-archive-queue"

  @BeanProperty
  @JsonProperty("in-process")
  var inProcess = new InProcess

  @BeanProperty
  @JsonProperty("backoff")
  var backOff = new {
    /**
     * How many times to try before giving up.
     *
     */
    @BeanProperty
    @JsonProperty("attempts")
    var backoffAttempts: Int = 25

    /**
     * What is the timeout (in ms) between retries.
     *
     */
    @BeanProperty
    @JsonProperty("timeout")
    var backoffTimeout: Int = 5000
  }

  @BeanProperty
  @JsonProperty("external")
  var external = new {
    /**
     * JMS Driver classname.
     * Examples: org.apache.activemq.ActiveMQConnectionFactory, com.rabbitmq.jms.admin.RMQConnectionFactory
     *
     */
    @BeanProperty
    @JsonProperty("jms-driver-classname")
    var jmsDriverClassname: String = _

    /**
     * JMS URL.
     *
     * Example: amqp://localhost:5672
     *
     */
    @BeanProperty
    @JsonProperty("jms-url")
    var jmsUrl: String = _

    /**
     * JMS username
     *
     */
    @BeanProperty
    @JsonProperty("jms-username")
    var jmsUsername: String = _

    /**
     * JMS password
     *
     */
    @BeanProperty
    @JsonProperty("jms-password")
    var jmsPassword: String = _

    /**
     * Specific configuration for Active MQ Artemis.
     */
    @BeanProperty
    @JsonProperty("artemis")
    var artemis = new {

      /**
       * Local bind address that the datagram socket is bound to
       *
       */
      @BeanProperty
      @JsonProperty("local-bind-address")
      var localBindAddress: String = _

      /**
       * Local port to which the datagram socket is bound to
       *
       */
      @BeanProperty
      @JsonProperty("local-bind-port")
      var localBindPortPath: Int = _

    }
  }
}

class SchedulerConfiguration {
  @BeanProperty
  @JsonProperty("system")
  var system = new SchedulerSystem
}

class SchedulerSystem {
  @BeanProperty
  @JsonProperty("akka")
  var akka = new SchedulerSystemAkka
}

@JsonInclude(Include.NON_EMPTY)
class SchedulerSystemAkka {
  /**
   * If true, scheduler will be running as a daemonic process.
   *
   */
  @BeanProperty
  @JsonProperty("daemonic")
  var daemonic: Boolean = true

  /**
   * Loggers to register at boot time.
   *
   */
  @BeanProperty
  @JsonProperty("loggers")
  @DocDefaultValue(value="[\"akka.event.slf4j.Slf4jLogger\"]", removeQuote = true)
  var loggers: JList[String] = JList.of("akka.event.slf4j.Slf4jLogger")

  /**
   * Options: OFF, ERROR, WARNING, INFO, DEBUG.
   *
   */
  @BeanProperty
  @JsonProperty("loglevel")
  var loglevel: String = "INFO"

  @BeanProperty
  @JsonProperty("scheduler")
  var scheduler = new SchedulerSystemAkkaScheduler
}

class SchedulerSystemAkkaScheduler {
  /**
   * How often scheduler is triggered.
   *
   */
  @BeanProperty
  @JsonProperty("tick-duration")
  var tickDuration: String = "1000ms"
}

class PlannerConfiguration {
  @BeanProperty
  @JsonProperty("file-watch")
  var fileWatch = new FileWatchConfiguration

  @BeanProperty
  @JsonProperty("registries")
  var registries = new PlannerRegistries

  @BeanProperty
  @JsonProperty("system")
  var system = new PlannerSystem
}

class FileWatchConfiguration {
  /**
   * Interval, in which file watcher is triggered.
   *
   */
  @BeanProperty
  @JsonProperty("interval")
  var interval: String = "1 second"
}

class PlannerRegistries {
  @BeanProperty
  @JsonProperty("Dispatcher")
  var dispatcher = new PlannerRegistryDispatcher

  /**
   * Number of core threads ... ceil(available processors * factor)
   *
   */
  @BeanProperty
  @JsonProperty("timeout")
  var corePoolSizeFactor: String = "1 minute"
}

class PlannerRegistryDispatcher {
  /**
   * What kind of ExecutionService to use.
   *
   */
  @BeanProperty
  @JsonProperty("executor")
  var executor: String = "thread-pool-executor"

  @BeanProperty
  @JsonProperty("thread-pool-executor")
  var threadPoolExecutor = new PlannerRegistryDispatcherThread

  /**
   * Throughput defines the maximum number of messages to be processed per actor before the thread jumps to the next actor.
   * Set to 1 for as fair as possible.
   *
   */
  @BeanProperty
  @JsonProperty("throughput")
  var throughput: Int = 100

  /**
   * An Akka MessageDispatcher is what makes Akka Actors “tick”, it is the engine of the machine so to speak.
   * All MessageDispatcher implementations are also an Executor, which means that they can be used to execute arbitrary code,
   * for instance CompletableFutures.
   *
   * There are 2 different types of message dispatchers:
   *
   * 1. Dispatcher
   *   This is an event-based dispatcher that binds a set of Actors to a thread pool. The default dispatcher is used if no other is specified.
   * 2. PinnedDispatcher
   *   This dispatcher dedicates a unique thread for each actor using it;
   *   i.e. each actor will have its own thread pool with only one thread in the pool.
   *
   */
  @BeanProperty
  @JsonProperty("type")
  var dispatcherType: String = "Dispatcher"
}

class PlannerRegistryDispatcherThread {
  /**
   * Number of core threads ... ceil(available processors * factor)
   *
   */
  @BeanProperty
  @JsonProperty("core-pool-size-factor")
  var corePoolSizeFactor: Int = 2

  /**
   * Maximum number of threads to cap factor-based number to.
   *
   */
  @BeanProperty
  @JsonProperty("core-pool-size-max")
  var corePoolSizeMax: Int = 10

  /**
   * Minimum number of threads to cap factor-based core number to.
   *
   */
  @BeanProperty
  @JsonProperty("core-pool-size-min")
  var corePoolSizeMin: Int = 10
}

class PlannerSystem {
  @BeanProperty
  @JsonProperty("akka")
  var akka = new PlannerSystemAkka
}

class PlannerSystemAkka {
  /**
   * Loggers to register at boot time.
   *
   */
  @BeanProperty
  @JsonProperty("loggers")
  @DocDefaultValue(value="[\"akka.event.slf4j.Slf4jLogger\"]", removeQuote = true)
  var loggers: JList[String] = JList.of("akka.event.slf4j.Slf4jLogger")

  /**
   * Options: OFF, ERROR, WARNING, INFO, DEBUG.
   *
   */
  @BeanProperty
  @JsonProperty("loglevel")
  var loglevel: String = "INFO"
}

sealed abstract class TaskQueueBrokerConfig() {}

case class InProcessConfig(maxDiskUsage: Int, shutdownTimeout: Long) extends TaskQueueBrokerConfig

case class ExternalConfig(jmsDriverClassname: String, url: String, username: String, password: String) extends TaskQueueBrokerConfig
