package ai.digital.configuration.central.deploy

import ai.digital.doc.properties.{DocDefaultValue, DocPropertiesFile}
import com.fasterxml.jackson.annotation.JsonInclude.Include
import com.fasterxml.jackson.annotation.{JsonInclude, JsonProperty}
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

import java.beans.BeanProperty
import java.util.{List => JList}

/**
  * Configuration for active/hot-standby mode
  */
@Configuration
@ConfigurationProperties(prefix = "deploy.cluster")
@DocPropertiesFile("deploy-cluster.yaml")
class ClusterProperties {

  /**
    * There are 2 options:
    * * default
    * * hot-standby
    *
    * To enabled hot standby mode, this value has to be equal to "hot-standby"
    *
    * Default: default
    */
  @BeanProperty
  @JsonProperty("mode")
  var mode: String = "default"

  /**
    * The given name for a cluster.
    * It will have an suffix in the akka logging for tracking.
    *
    * Default: xld-hotstandby-cluster
    */
  @BeanProperty
  @JsonProperty("name")
  var name: String = "xld-hotstandby-cluster"

  /**
    * The core of Akka Cluster is the cluster membership, to keep track of what nodes are part of the cluster
    * and their health. Cluster membership is communicated using gossip and failure detection.
    */
  @BeanProperty
  @JsonProperty("membership")
  var membership: Membership = new Membership

  /**
    * Other akka specific cluster configurations
    */
  @BeanProperty
  @JsonProperty("akka")
  var akka: Akka = new Akka

}

class Membership {

  /**
    * How often a node should write liveness information into the database.
    *
    * When using Akka Cluster this is the important failure detector,
    * and you should normally not worry about the other two failure detectors.
    * Each node in an Akka Cluster monitors a few other nodes by sending heartbeat messages to them
    * and expecting timely response messages.
    *
    * If no heartbeat replies are received within a timeout the node
    * is marked as unreachable. A node marked as unreachable will become reachable again
    * if the failure detector observes that it can communicate with it again,
    * i.e. unreachable is not a fatal condition.
    *
    * Default: 10 seconds
    */
  @BeanProperty
  @JsonProperty("heartbeat")
  var heartbeat: String = "10 seconds"

  /**
    * How long liveness information remains valid.
    *
    * Default: 60 seconds
    */
  @BeanProperty
  @JsonProperty("ttl")
  var ttl: String = "60 seconds"

  /**
    * When active/hot-standby is enabled, the external repository database must be shared among all nodes.
    * Ensure that each node has access to the shared repository database and
    * is set up to connect to it using the same credentials.
    *
    * If you are storing artifacts in a separate filesystem location, ensure that it is shared among the Deploy nodes;
    * for example, by using NFS or by mounting the same volume.
    */
  @BeanProperty
  @JsonProperty("jdbc")
  var jdbc: JdbcConfig = new JdbcConfig
}

@JsonInclude(Include.NON_NULL)
class JdbcConfig {
  /**
    * Database URL for active/hot-standby setup
    */
  @BeanProperty
  @JsonProperty("url")
  var url: String = _

  /**
    * User name to use when logging into the database.
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("username")
  var username: String = _

  /**
    * Password to use when logging into the database.
    * After you complete the setup, the password will be encrypted and stored in a secured format.
    *
    * Default: none
    */
  @BeanProperty
  @JsonProperty("password")
  var password: String = _

  /**
    * This property represents a user-defined name for the connection pool and appears mainly in logging
    * and JMX management consoles to identify pools and pool configurations.
    *
    * Default: ClusterPool
    */
  @BeanProperty
  @JsonProperty("pool-name")
  var poolName: String = "ClusterPool"

  /**
    * This property controls the maximum size that the pool is allowed to reach,
    * including both idle and in-use connections.
    * Basically this value will determine the maximum number of actual connections to the database backend.
    * A reasonable value for this is best determined by your execution environment.
    * When the pool reaches this size, and no idle connections are available, calls to getConnection()
    * will block for up to connectionTimeout milliseconds before timing out.
    * Please read about pool sizing.
    *
    * Default: 1
    */
  @BeanProperty
  @JsonProperty("max-pool-size")
  var maxPoolSize: Int = 1

  /**
    * This property controls the minimum number of idle connections that HikariCP tries to maintain in the pool.
    * If the idle connections dip below this value and total connections in the pool are less than maximumPoolSize,
    * HikariCP will make a best effort to add additional connections quickly and efficiently.
    * However, for maximum performance and responsiveness to spike demands,
    * we recommend not setting this value and instead allowing HikariCP to act as a fixed size connection pool.
    *
    * Default: 1
    */
  @BeanProperty
  @JsonProperty("minimum-idle")
  var minimumIdle: Int = 1

  /**
    * This property controls the maximum number of milliseconds that a client (that's you) will wait for a
    * connection from the pool. If this time is exceeded without a connection becoming available,
    * a SQLException will be thrown. Lowest acceptable connection timeout is 250 ms.
    *
    * Default: 30 seconds
    */
  @BeanProperty
  @JsonProperty("connection-timeout")
  var connectionTimeout: String = "30 seconds"

  /**
    * This property controls the maximum lifetime of a connection in the pool. An in-use connection will never be retired,
    * only when it is closed will it then be removed. On a connection-by-connection basis, minor negative attenuation
    * is applied to avoid mass-extinction in the pool.
    * We strongly recommend setting this value, and it should be several seconds shorter than any database or
    * infrastructure imposed connection time limit. A value of 0 indicates no maximum lifetime (infinite lifetime),
    * subject of course to the idleTimeout setting. The minimum allowed value is 30 seconds.
    *
    * Default: 30 minutes
    */
  @BeanProperty
  @JsonProperty("max-life-time")
  var maxLifeTime: String = "30 minutes"

  /**
    * This property controls the maximum amount of time that a connection is allowed to sit idle in the pool.
    * This setting only applies when minimumIdle is defined to be less than maximumPoolSize.
    * Idle connections will not be retired once the pool reaches minimumIdle connections.
    * Whether a connection is retired as idle or not is subject to a maximum variation of +30 seconds,
    * and average variation of +15 seconds. A connection will never be retired as idle before this timeout.
    * A value of 0 means that idle connections are never removed from the pool.
    * The minimum allowed value is 10 seconds.
    *
    * Default: 10 minutes
    */
  @BeanProperty
  @JsonProperty("idle-timeout")
  var idleTimeout: String = "10 minutes"

  /**
    * This property controls the amount of time that a connection can be out of the pool before a message is
    * logged indicating a possible connection leak. A value of 0 means leak detection is disabled.
    * Lowest acceptable value for enabling leak detection is 2 seconds.
    *
    * Default: 15 seconds
    */
  @BeanProperty
  @JsonProperty("leak-connection-threshold")
  var leakConnectionThreshold: String = "15 seconds"
}


class Akka {
  @BeanProperty
  @JsonProperty("actor")
  var actor: Actor = new Actor

  @BeanProperty
  @JsonProperty("remote")
  var remote: Remote = new Remote

  @BeanProperty
  @JsonProperty("cluster")
  var cluster: cluster = new cluster
}

class Actor {

  /**
    * The ClusterActorRefProvider will load the Cluster extension, i.e. the cluster will automatically
    * be started when the ClusterActorRefProvider is used.
    *
    * Default: akka.cluster.ClusterActorRefProvider
    */
  @BeanProperty
  @JsonProperty("provider")
  var provider: String = "akka.cluster.ClusterActorRefProvider"

  /**
    * Loggers to register at boot time.
    *
    * Default: ["akka.event.slf4j.Slf4jLogger"]
    */
  @BeanProperty
  @JsonProperty("loggers")
  @DocDefaultValue(value="[\"akka.event.slf4j.Slf4jLogger\"]", removeQuote=true)
  var loggers: JList[String] = JList.of("akka.event.slf4j.Slf4jLogger")

  /**
    * Options: OFF, ERROR, WARNING, INFO, DEBUG.
    *
    * Default: INFO
    */
  @BeanProperty
  @JsonProperty("loglevel")
  var loglevel: String = "INFO"

}

class Remote {
  @BeanProperty
  @JsonProperty("artery")
  var artery: Artery = new Artery
}

class Artery {
  @BeanProperty
  @JsonProperty("canonical")
  var canonical: Canonical = new Canonical
}

@JsonInclude(Include.NON_NULL)
class Canonical {
  /**
    * The hostname or IP that Deploy uses to communicate with the cluster.
    *
    * Default: none (Auto-determined)
    */
  @BeanProperty
  @JsonProperty("hostname")
  var hostname: String = _

  /**
    * The port number that Deploy uses to communicate with the cluster.
    *
    * Default: 25520
    */
  @BeanProperty
  @JsonProperty("port")
  var port: Int = 25520
}

@JsonInclude(Include.NON_NULL)
class cluster {

  /**
    * The strategy to use for handling network partitions.
    * Possible options are akka.cluster.sbr.SplitBrainResolverProvider,
    * com.xebialabs.xlplatform.cluster.full.downing.MajorityLeaderAutoDowningProvider,
    * com.xebialabs.xlplatform.cluster.full.downing.OldestLeaderAutoDowningProvider and
    * akka.cluster.NoDowning.
    *
    * Default: akka.cluster.sbr.SplitBrainResolverProvider
    */
  @BeanProperty
  @JsonProperty("downing-provider-class")
  var downingProviderClass: String = "akka.cluster.sbr.SplitBrainResolverProvider"

  @BeanProperty
  @JsonProperty("custom-downing")
  var customDowning: CustomDowning = new CustomDowning
}

class CustomDowning {
  /**
    * How much time must pass before the network is marked stable.
    *
    * Default: 10s
    */
  @BeanProperty
  @JsonProperty("stable-after")
  var stableAfter: String = "10s"

  /**
    * The amount of time before the handover activates.
    *
    * Default: 10s
    */
  @BeanProperty
  @JsonProperty("down-removal-margin")
  var downRemovalMargin: String = "10s"
}
