package ai.digital.configuration.central.deploy

import ai.digital.doc.properties.DocPropertiesFile
import com.fasterxml.jackson.annotation.JsonInclude.Include
import com.fasterxml.jackson.annotation.{JsonInclude, JsonProperty}
import org.springframework.boot.context.properties.ConfigurationProperties
import org.springframework.context.annotation.Configuration

import java.beans.BeanProperty

/**
  * This Class is used to configure Monitoring System which uses InfluxDbFileReporter,
  *  to report values in the  metric registry to InfluxDB ( time series database) readable files.
  *
  *  Collected file data has the internal and system metrics over Java Management Extensions (JMX).
  *  Enabled Monitoring System, can read JMX data with our products.
  *  Captured data will look like below.
  *  {{{
  *    jdbc,host=XLD-01,instance=Statement,type=Durations,type_instance=SELECT$1XLPM_PLUGIN.ID$1$1NAME$1$1VERSION$1
  * $1$1$1$1$1$1$1$1EXTENSION$1$1GROUP_ID$1$1INSTALLATION_STATUS$1$1SOURCE$1
  * $1$1$1$1$1$1$1$1CHECKSUM$1$1$1PLUGIN_BYTES_ID$1$1BYTES
  * $1$1$1$1$1$1$1$1FROM$1XLPM_PLUGIN$1$1XLPM_PLUGIN_BYTES
  * $1$1$1$1$1$1$1$1WHERE$1XLPM_PLUGIN.PLUGIN_BYTES_ID$1$1$1XLPM_PLUGIN_BYTES.ID value=9013715i 1623726432000000000
  * jdbc,host=XLD-01,instance=Statement,type=Durations,type_instance=UPDATE$1xld.DATABASECHANGELOGLOCK$1SET$1`LOCKED`$1$1$11$1$1LOCKEDBY$1$1$1'172.19.0.1$1(172.19.0.1)'$1$1LOCKGRANTED$1$1$1'2021-06-15$108:36:23.963'$1WHERE$1ID$1$1$11$1AND$1`LOCKED`$1$1$10 value=634946i 1623726432000000000
  * jdbc,host=XLD-01,instance=Statement,type=Durations,type_instance=update$1`XLD_WORKERS`$1set$1`public_key`$1$1$1?$1$1`configuration_hash`$1$1$1?$1$1`name`$1$1$1?$1where$1`address`$1$1$1? value=771207i 1623726432000000000
  *  }}}
  */
@Configuration
@ConfigurationProperties(prefix = "deploy.metrics")
@JsonInclude(Include.NON_NULL)
@DocPropertiesFile("deploy-metrics.yaml")
class MetricsProperties {

  /**
    * Property to enable the use of monitoring systems that can read JMX data.
    * Default to false
    */
  @BeanProperty
  @JsonProperty("enabled")
  var enabled: Boolean = false

  /**
    * Name of the JVM instance or machine that generates the metrics,
    * which is used to identify the source of the metrics in InfluxDB.
    * Default to XLD
    * {{{
    *   Captured data will look like "jdbc,host=XLD-01,instance=Statement,type=Durations,type_instance=update$1`XLD_WORKERS`$1set$1`public_key`$1$1$1?$1$1`configuration_hash`$1$1$1?$1$1`name`$1$1$1?$1where$1`address`$1$1$1? value=771207i 1623726432000000000"
    *   here instance name is configured as "XLD-01"
    * }}}
    */
  @BeanProperty
  @JsonProperty("instance-name")
  var instanceName = "XLD"

  /**
    * Property that specifies the directory of the metric report file.
    * Default to metrics.
    * {{{
    *   for example - if we configure path to be "metricsReport", it will create the metricsReport folder in the current directory.
    * }}}
    */
  @BeanProperty
  @JsonProperty("path")
  var path = "metrics"

  /**
    * Property that specifies the file name format (dateFormat specified). Set as "yyyyMMdd" to default
    * {{{
    *   for example - "yyyyMMdd", then the generated file name would be "20210615.txt"
    * }}}
    */
  @BeanProperty
  @JsonProperty("date-format")
  var dateFormat = "yyyyMMdd"

  /**
    * Property that specifies the maximum number of files allowed in the directory. Default to 7
    * {{{
    *  for example - maxFileCount is 7
    *   Metrics folder(Path to the report)  can contain maximum of recent 7 files in .gz format
    *   and whenever there is a new file created, max count is validated, if it is >= maxCount,
    *   oldest file will be deleted and new file gets added.
    * }}}
    */
  @BeanProperty
  @JsonProperty("max-file-count")
  var maxFileCount: Int = 7

  /**
    * Property to define the time interval for report generation in seconds. Default to 10 secs
    */
  @BeanProperty
  @JsonProperty("report-interval-secs")
  var reportIntervalSecs: Int = 10

}
