package ai.digital.config.server

import ai.digital.config.{ConfigClientSystemValues, RetryConfigClientSystemValues, ServerConfigurationHelper}
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.util.PasswordEncrypter
import grizzled.slf4j.Logging

trait StandaloneConfigServerSystemValues extends RetryConfigClientSystemValues with ConfigClientSystemValues with Logging {

  def applyConfigServerSystemValues(): this.type = {
    System.setProperty("spring.config.name", "central-configuration")
    applyConfigClientOnlySystemValues()
  }

  def applyConfigServerSystemValues(serverConfiguration: ServerConfiguration): this.type = {
    System.setProperty("spring.security.user.name", ServerConfigurationHelper.ConfigAdminUser)
    // Use the default password. Allowing to change the password will require change from clients to have the password on their side
    System.setProperty("spring.security.user.password", decrypt(ServerConfiguration.DEFAULT_ADMIN_PASSWORD))
    // encrypt.key needs to be same also on client side that is updating secure content
    System.setProperty("encrypt.key", serverConfiguration.getSpringCloudEncryptKey)

    setConfigIfPresent("server.port", serverConfiguration.getHttpPort.toString)

    //ssl settings
    setSslConfig(serverConfiguration)
    applyConfigClientRetrySystemValues(serverConfiguration)
  }

  private def setSslConfig(serverConfiguration: ServerConfiguration): Unit = {
    if (serverConfiguration.isSsl) {
      setConfigIfPresent("server.url", serverConfiguration.getServerUrl)
      setConfigIfPresent("server.ssl.enabled", serverConfiguration.isSsl.toString)
      setConfigIfPresent("server.ssl.key-store", serverConfiguration.getKeyStorePath)
      setConfigIfPresent("server.ssl.key-store-type", serverConfiguration.getKeyStoreType)
      setConfigIfPresent("server.ssl.key-store-password", decrypt(serverConfiguration.getKeyStorePassword))
      setConfigIfPresent("server.ssl.key-password", decrypt(serverConfiguration.getKeyStoreKeyPassword))
      setConfigIfPresent("server.ssl.trust-store", serverConfiguration.getTrustStorePath)
      setConfigIfPresent("server.ssl.trust-store-password", decrypt(serverConfiguration.getTrustStorePassword))
    }
  }

  private def setConfigIfPresent(key : String, configValue: AnyRef) : Unit = {
    Option(configValue) match {
      case Some(value) => System.setProperty(key, value.asInstanceOf[String])
      case None => logger.info(s"Property  $key is not set in deployit.conf so skipping")
    }
  }

  private def decrypt(value: String): String = PasswordEncrypter.getInstance().ensureDecrypted(value)
}
