package ai.digital.config.server.encryptor

import ai.digital.config.server.api.{ConfigurationPropertiesService, PropertyValue}
import ai.digital.config.server.client.LocalConfigurationReader
import ai.digital.config.{ConfigFileFilter, ConfigFileMapping, ContainsPropertyKeyFilter, PropertyKeyFilter}
import org.springframework.beans.factory.annotation.Qualifier
import grizzled.slf4j.Logging
import org.springframework.boot.context.event.ApplicationReadyEvent
import org.springframework.context.event.EventListener

class CentralConfigEncryptor(val configurationReader: LocalConfigurationReader,
                             val textEncryptor: CentralConfigTextEncryptor,
                             @Qualifier("defaultConfigurationPropertiesService")
                             val configurationPropertiesService: ConfigurationPropertiesService,
                             val configFileFilter: ConfigFileFilter,
                             val propertyKeyFilters: List[PropertyKeyFilter] = List(new ContainsPropertyKeyFilter("password")))
 extends Logging {

  @EventListener(Array(classOf[ApplicationReadyEvent]))
  def run(): Unit = {
    val sources =
      configurationReader.findPropertySources()
        .filter { case (source, _) => source.getName.endsWith(".yaml") }
        .map{ case (source, properties) =>
          (
            source,
            properties.filter(entry => propertyKeyFilters.exists(_.shouldBeEncrypted(entry._1)))
          )
        }.filter(_._2.nonEmpty)

    val properties = sources
      .flatMap { case (_, entries) =>
        entries
          .filter { case (_, value) => !textEncryptor.isEncrypted(value.toString) }
          .map { case (key, value) => (key, textEncryptor.encrypt(value.toString)) }
      }
      .map { case (key, value) => (key, PropertyValue(value, null, false)) }

    logger.info(s"Encrypting following properties ${properties.map(_._1).mkString(", ")}")

    configurationPropertiesService.propertiesUpdate(
      configFileFilter,
      properties.toMap,
      ConfigFileMapping()
    )
  }
}
