package ai.digital.config

import com.xebialabs.deployit.ServerConfiguration
import grizzled.slf4j.Logging
import org.springframework.boot.{ConfigurableBootstrapContext, SpringApplicationRunListener}
import org.springframework.core.env.{ConfigurableEnvironment, MapPropertySource}

import scala.jdk.CollectionConverters._

abstract class AbstractEnvironmentPreparedProcessor extends SpringApplicationRunListener with Logging {

  override def environmentPrepared(bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Unit =
    addPropertySource(springCloudEnvironment(bootstrapContext, environment), bootstrapContext, environment)

  protected def springCloudEnvironment(bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Map[String, AnyRef] =
    Map[String, Option[AnyRef]](
      "spring.cloud.config.uri" -> resolveSpringCloudUri(bootstrapContext, environment),
      "spring.cloud.config.username" -> resolveSpringCloudConfigUsername(bootstrapContext, environment),
      "spring.cloud.config.password" -> resolveSpringCloudConfigPassword(bootstrapContext, environment))
      .flatMap { case (k, v) => v.map((k, _)) }

  protected def addPropertySource(env: Map[String, AnyRef], bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Unit =
    if (env.nonEmpty) {
      val propertySource = new MapPropertySource(propertyResourceName, env.asJava)
      environment.getPropertySources.addFirst(propertySource)
    }

  protected def propertyResourceName: String

  protected def resolveSpringCloudUri(bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Option[String] =
    Option(ServerConfiguration.getInstance().getSpringCloudUri)

  protected def resolveSpringCloudConfigUsername(bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Option[String] =
    Some(ServerConfigurationHelper.ConfigAdminUser)

  protected def resolveSpringCloudConfigPassword(bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Option[String] =
    ServerConfigurationHelper.getAdminPassword
}
