package ai.digital.config

import grizzled.slf4j.Logging
import org.apache.commons.lang.StringEscapeUtils
import org.springframework.boot.env.PropertiesPropertySourceLoader
import org.springframework.core.Ordered
import org.springframework.core.annotation.Order
import org.springframework.core.env.PropertySource
import org.springframework.core.io.Resource

import java.io.{FileWriter, PrintWriter}
import java.util
import scala.collection.mutable
import scala.collection.mutable.ListBuffer
import scala.util.Using

@Order(Ordered.HIGHEST_PRECEDENCE)
class EscapingPropertiesPropertySourceLoader extends PropertiesPropertySourceLoader with Logging {
  override def load(name: String, resource: Resource): util.List[PropertySource[_]] = {
    logger.debug("inside load of EscapingPropertiesPropertySourceLoader")
    updatePropertiesFileWithEscapedValues(resource)
    super.load(name, resource)
  }

  private def updatePropertiesFileWithEscapedValues(resource: Resource) = {
    val lines = mutable.ListBuffer.empty[String]
    Using(scala.io.Source.fromFile(resource.getFile)) {
      source =>
        source
          .getLines()
          .foreach(line => processAndEscapeLine(line).foreach(lines.addOne))
    }
    if (!lines.isEmpty) {
      writeToFile(resource, lines)
    }
  }

  private def processAndEscapeLine(line: String): Option[String] = {
    if (!line.startsWith("#") && line.contains("=") && !line.contains("\\\\")) {
      val i = line.indexOf("=")
      val key = line.substring(0, i)
      val value = line.substring(i + 1)
      //TODO: the below code escapes already escaped '\'. Need to find a way to do that only once.
      Some(s"$key=${StringEscapeUtils.escapeJava(value)}")
    } else Some(line)
  }

  private def writeToFile(resource: Resource, lines: ListBuffer[String]) = {
    Using(new PrintWriter(new FileWriter(resource.getFile, false))) {
      writer => lines.foreach(line => writer.println(line))
    }
  }

}
