package ai.digital.config.server.client

import ai.digital.config.{ConfigFileFilter, ConfigServerProfiles}
import ai.digital.config.server.api.ConfigurationPropertiesService
import com.fasterxml.jackson.databind.ObjectMapper
import com.xebialabs.deployit.ServerConfiguration
import grizzled.slf4j.Logging
import org.apache.http.impl.client.CloseableHttpClient
import org.springframework.beans.factory.annotation.{Qualifier, Value}
import org.springframework.boot.autoconfigure.condition.{ConditionalOnClass, ConditionalOnMissingBean}
import org.springframework.cloud.context.refresh.ContextRefresher
import org.springframework.context.annotation.{Bean, ComponentScan, Configuration, Primary, Profile}
import org.springframework.scheduling.annotation.EnableScheduling

import javax.annotation.PostConstruct

@Configuration
@ConditionalOnClass(Array(classOf[CloseableHttpClient], classOf[ContextRefresher], classOf[ServerConfiguration], classOf[ObjectMapper]))
@EnableScheduling
@ComponentScan
class CentralConfigurationClientAutoconfiguration(@Value("${spring.application.name:central-config-client}") val applicationName: String,
                                                  @Value("${spring.profiles.active:}") val activeProfiles: String,
                                                  @Value("${spring.cloud.config.uri:}") springCloudUri: String)
  extends Logging {

  @PostConstruct
  def startClient(): Unit = {
    Option(springCloudUri)
      .map(uri => s"Starting central configuration client pointing to ${springCloudUri}")
      .getOrElse(s"Starting central configuration client pointing to embedded server")
  }

  @Bean
  @ConditionalOnMissingBean(Array(classOf[ConfigFileFilter]))
  def configFileFilter(): ConfigFileFilter =
    new ConfigFileFilter(applicationName.replace(" ", ""), activeProfiles.replace(" ", ""))

  @Bean
  @ConditionalOnMissingBean(Array(classOf[ServerConfiguration]))
  def serverConfiguration(): ServerConfiguration =
    Option(ServerConfiguration.getInstance())
      .getOrElse(throw new IllegalStateException("Server configuration is not configured before spring configuration"))

  @Profile(Array(ConfigServerProfiles.CONFIG_SERVER))
  @ConditionalOnClass(name = Array("ai.digital.config.server.api.EnvironmentConfigurationPropertiesService"))
  @Primary
  @Bean(Array("defaultConfigurationPropertiesService"))
  def environmentConfigurationPropertiesService(@Qualifier("environmentConfigurationPropertiesService")
                                                environmentConfigurationPropertiesService: ConfigurationPropertiesService): ConfigurationPropertiesService = {
    logger.info("Using local central configuration updates")
    environmentConfigurationPropertiesService
  }

  @Profile(Array(ConfigServerProfiles.NOT_CONFIG_SERVER))
  @Primary
  @Bean(Array("defaultConfigurationPropertiesService"))
  def remoteConfigurationPropertiesService(@Qualifier("remoteConfigurationPropertiesService")
                                           remoteConfigurationPropertiesService: ConfigurationPropertiesService): ConfigurationPropertiesService = {
    logger.info("Using remote central configuration updates")
    remoteConfigurationPropertiesService
  }
}
