package ai.digital.config.server.client

import ai.digital.config.server.api._
import ai.digital.config.server.client.HttpClientHelper.{HttpDeleteWithBody, HttpPutWithBody}
import ai.digital.config.{ConfigFileFilter, ConfigFileMapping}
import com.fasterxml.jackson.databind.ObjectMapper
import grizzled.slf4j.Logging
import org.apache.http.HttpHeaders
import org.apache.http.client.methods.HttpPost
import org.apache.http.entity.{ContentType, StringEntity}
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class RemoteConfigurationPropertiesService(val httpClient: HttpClientHelper, val objectMapper: ObjectMapper) extends ConfigurationPropertiesService with Logging {

  override def propertiesStore(configFileFilter: ConfigFileFilter,
                               header: String,
                               properties: Iterable[(String, PropertyValue)],
                               configFileMapping: ConfigFileMapping): Iterable[String] =
    httpClient.withClient { client =>
      val httpPost = new HttpPost(httpClient.buildUri(toPath(configFileFilter)))
      val body = PropertiesStoreRequestBody(
        header = header,
        properties = properties.map{ case (key, value) => Property(key, value) }.asJava,
        configFileToPropertyKeysMapping = configFileMapping.configFileToPropertyKeysMapping.asJava,
        defaultConfigFile = configFileMapping.defaultConfigFile.orNull
      )
      httpPost.setEntity(new StringEntity(
        objectMapper
          .writerFor(classOf[PropertiesStoreRequestBody])
          .writeValueAsString(body)
      ))
      httpPost.setHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType)
      val response = httpClient.execute(client, httpPost)
      val updatedProperties = objectMapper.readValue(response, classOf[PropertiesStoreResponse])
      val keys = updatedProperties.getUpdatedKeys.asScala
      logger.debug(s"Stored properties keys $keys")
      keys
    }


  override def propertiesUpdate(configFileFilter: ConfigFileFilter,
                                properties: Map[String, PropertyValue],
                                configFileMapping: ConfigFileMapping): Iterable[String] =
    httpClient.withClient { client =>
      val httpPut = new HttpPutWithBody(httpClient.buildUri(toPath(configFileFilter)))
      val body = PropertiesUpdateRequestBody(
        properties = properties.asJava,
        configFileToPropertyKeysMapping = configFileMapping.configFileToPropertyKeysMapping.asJava,
        defaultConfigFile = configFileMapping.defaultConfigFile.orNull
      )
      httpPut.setEntity(new StringEntity(
        objectMapper
          .writerFor(classOf[PropertiesUpdateRequestBody])
          .writeValueAsString(body)
      ))
      httpPut.setHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType)
      val response = httpClient.execute(client, httpPut)
      val updatedProperties = objectMapper.readValue(response, classOf[PropertiesUpdateResponse])
      val keys = updatedProperties.getUpdatedKeys.asScala
      logger.debug(s"Updated properties keys $keys")
      keys
    }

  override def propertiesDelete(configFileFilter: ConfigFileFilter, propertyKeys: Set[String], configFileMapping: ConfigFileMapping): Iterable[String] =
    httpClient.withClient { client =>
      val httpDelete = new HttpDeleteWithBody(httpClient.buildUri(toPath(configFileFilter)))
      val body = PropertiesDeleteRequestBody(
        keys = propertyKeys.asJava,
        configFileToPropertyKeysMapping = configFileMapping.configFileToPropertyKeysMapping.asJava,
        defaultConfigFile = configFileMapping.defaultConfigFile.orNull
      )
      httpDelete.setEntity(new StringEntity(
        objectMapper
          .writerFor(classOf[PropertiesDeleteRequestBody])
          .writeValueAsString(body)
      ))
      httpDelete.setHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType)
      val response = httpClient.execute(client, httpDelete)
      val deletedProperties = objectMapper.readValue(response, classOf[PropertiesDeleteResponse])
      val keys = deletedProperties.getDeletedKeys.asScala
      logger.debug(s"Deleted properties $keys")
      keys
    }

  def toPath(configFileFilter: ConfigFileFilter): String =
    Seq(configFileFilter.application, configFileFilter.profile, configFileFilter.label)
      .filter(Option(_).isDefined)
      .mkString("/")
}
