package ai.digital.config.server.client

import ai.digital.config.ServerConfigurationHelper
import com.xebialabs.deployit.ServerConfiguration
import org.apache.commons.codec.binary.Base64
import org.apache.http.HttpHeaders
import org.apache.http.client.config.RequestConfig
import org.apache.http.client.methods.{HttpDelete, HttpPut, HttpUriRequest}
import org.apache.http.conn.ssl.NoopHostnameVerifier
import org.apache.http.impl.client.{CloseableHttpClient, HttpClientBuilder, HttpClients, SystemDefaultCredentialsProvider}
import org.apache.http.impl.conn.SystemDefaultRoutePlanner
import org.apache.http.ssl.SSLContextBuilder
import org.apache.http.util.EntityUtils
import org.jboss.resteasy.util.HttpHeaderNames
import org.springframework.beans.factory.annotation.Value
import org.springframework.http.{HttpStatus, MediaType}
import org.springframework.stereotype.Component

import java.io.File
import java.net.{ProxySelector, URI}
import java.nio.charset.StandardCharsets
import java.security.cert.X509Certificate
import java.time.Duration

object HttpClientHelper {
  val actuatorRefreshPath = "/actuator/refresh"

  import org.apache.http.client.methods.HttpEntityEnclosingRequestBase

  class HttpPutWithBody(uri: URI) extends HttpEntityEnclosingRequestBase {
    setURI(uri)
    setHeader(HttpHeaderNames.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
    setHeader(HttpHeaderNames.ACCEPT, MediaType.APPLICATION_JSON_VALUE)

    override def getMethod: String = HttpPut.METHOD_NAME
  }

  class HttpDeleteWithBody(uri: URI) extends HttpEntityEnclosingRequestBase {
    setURI(uri)
    setHeader(HttpHeaderNames.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
    setHeader(HttpHeaderNames.ACCEPT, MediaType.APPLICATION_JSON_VALUE)

    override def getMethod: String = HttpDelete.METHOD_NAME
  }
}

@Component
class HttpClientHelper(@Value("${spring.cloud.config.uri:}") springCloudUri: String, serverConfiguration: ServerConfiguration) {

  val authHeader: Option[String] = {
    ServerConfigurationHelper.getAdminPassword.map { password =>
      val auth = s"${ServerConfigurationHelper.getAdminUser}:$password"
      val encodedAuth = Base64.encodeBase64String(auth.getBytes(StandardCharsets.UTF_8))
      "Basic " + encodedAuth
    }
  }

  def execute(httpClient: CloseableHttpClient, request: HttpUriRequest): String = {
    authHeader.foreach(headerValue => request.setHeader(HttpHeaders.AUTHORIZATION, headerValue))
    val httpResponse = httpClient.execute(request)
    if (httpResponse.getEntity != null) {
      if (httpResponse.getStatusLine.getStatusCode != HttpStatus.OK.value()) {
        val responseError = EntityUtils.toString(httpResponse.getEntity)
        throw new IllegalStateException(s"Central configuration request failed with status code ${httpResponse.getStatusLine.getStatusCode}: $responseError")
      }
      EntityUtils.toString(httpResponse.getEntity)
    } else {
      throw new IllegalStateException("Central configuration request failed")
    }
  }

  def buildUri(path: String, replace: String = ""): URI = {
    val slash = if (springCloudUri.endsWith("/") || path.startsWith("/")) "" else "/"
    URI.create(s"${springCloudUri}$slash$path".replace(replace, ""))
  }

  def withClient[R](block: CloseableHttpClient => R): R = {
    val client = builder().build()
    try {
      block(client)
    } finally {
      client.close()
    }
  }

  def builder(): HttpClientBuilder = {
    val sslContextBuilder = new SSLContextBuilder
    val httpClientBuilder = HttpClients.custom
    val timeout = Duration.ofMinutes(1).toMillis.toInt

    if (serverConfiguration.isSsl) {
      Option(serverConfiguration.getKeyStorePath)
        .map(path =>
          sslContextBuilder.loadTrustMaterial(
            new File(path), ServerConfigurationHelper.getKeyStorePassword.map(_.toCharArray).orNull)
        )
        .getOrElse(sslContextBuilder.loadTrustMaterial(null, (_: Array[X509Certificate], _: String) => true))
    } else {
      sslContextBuilder.loadTrustMaterial(null, (_: Array[X509Certificate], _: String) => true)
    }
    httpClientBuilder.setSSLHostnameVerifier(new NoopHostnameVerifier)
    httpClientBuilder.setRoutePlanner(new SystemDefaultRoutePlanner(ProxySelector.getDefault))
    httpClientBuilder.setDefaultCredentialsProvider(new SystemDefaultCredentialsProvider)
    httpClientBuilder.setSSLContext(sslContextBuilder.build)
      .setDefaultRequestConfig(RequestConfig.custom.setSocketTimeout(timeout).setConnectTimeout(timeout).build)
  }
}
