package ai.digital.config.server.client

import ai.digital.config.server.api._
import ai.digital.config.{ConfigFileFilter, ConfigFileMapping}
import grizzled.slf4j.Logging
import org.springframework.http.{HttpMethod, MediaType}
import org.springframework.retry.annotation.Retryable
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class RemoteConfigurationPropertiesService(val restTemplateHelper: RestTemplateHelper) extends ConfigurationPropertiesService with Logging {

  @Retryable(interceptor = "configServerRetryInterceptor")
  override def propertiesStore(configFileFilter: ConfigFileFilter,
                               header: String,
                               properties: Iterable[(String, PropertyValue)],
                               configFileMapping: ConfigFileMapping): Iterable[String] = {
    val body = PropertiesStoreRequestBody(
      header = header,
      properties = properties.map { case (key, value) => Property(key, value)}.asJava,
      configFileToPropertyKeysMapping = configFileMapping.configFileToPropertyKeysMapping.asJava,
      defaultConfigFile = configFileMapping.defaultConfigFile.orNull
    )

    val updatedProperties: Option[PropertiesStoreResponse] =
      restTemplateHelper.execute(toPath(configFileFilter),
        HttpMethod.POST,
        classOf[PropertiesStoreResponse],
        Some(body),
        mediaTypeMaybe = Some(MediaType.APPLICATION_JSON_VALUE))

    val keys = updatedProperties.map(_.getUpdatedKeys.asScala).getOrElse(Iterable.empty)
    logger.debug(s"Stored properties keys $keys")
    keys
  }

  @Retryable(interceptor = "configServerRetryInterceptor")
  override def propertiesUpdate(configFileFilter: ConfigFileFilter,
                                properties: Map[String, PropertyValue],
                                configFileMapping: ConfigFileMapping): Iterable[String] = {
    val body = PropertiesUpdateRequestBody(
      properties = properties.asJava,
      configFileToPropertyKeysMapping = configFileMapping.configFileToPropertyKeysMapping.asJava,
      defaultConfigFile = configFileMapping.defaultConfigFile.orNull
    )

    val updatedProperties: Option[PropertiesStoreResponse] =
      restTemplateHelper.execute(
        toPath(configFileFilter),
        HttpMethod.PUT,
        classOf[PropertiesStoreResponse],
        Some(body),
        mediaTypeMaybe = Some(MediaType.APPLICATION_JSON_VALUE))

    val keys = updatedProperties.map(_.getUpdatedKeys.asScala).getOrElse(Iterable.empty)
    logger.debug(s"Updated properties keys $keys")
    keys
  }

  @Retryable(interceptor = "configServerRetryInterceptor")
  override def propertiesDelete(configFileFilter: ConfigFileFilter,
                                propertyKeys: Set[String],
                                configFileMapping: ConfigFileMapping): Iterable[String] = {
    val body = PropertiesDeleteRequestBody(
      keys = propertyKeys.asJava,
      configFileToPropertyKeysMapping = configFileMapping.configFileToPropertyKeysMapping.asJava,
      defaultConfigFile = configFileMapping.defaultConfigFile.orNull
    )

    val deletedProperties: Option[PropertiesDeleteResponse] =
      restTemplateHelper.execute(
        toPath(configFileFilter),
        HttpMethod.DELETE,
        classOf[PropertiesDeleteResponse],
        Some(body),
        mediaTypeMaybe = Some(MediaType.APPLICATION_JSON_VALUE))

    val keys = deletedProperties.map(_.getDeletedKeys.asScala).getOrElse(Iterable.empty)
    logger.debug(s"Deleted properties $keys")
    keys
  }

  def toPath(configFileFilter: ConfigFileFilter): String =
    Seq(configFileFilter.application, configFileFilter.profile, configFileFilter.label)
      .filter(Option(_).isDefined)
      .mkString("/")
}
